<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\ConsoleLog;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class ConsoleLogController extends Controller
{
    /**
     * Armazena um novo log do console
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'type' => 'required|string|max:50',
            'message' => 'required|string',
            'data' => 'nullable|string',
            'severity' => 'required|string|in:error,warning,info',
            'context' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Dados inválidos',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $log = ConsoleLog::create([
                'type' => $request->type,
                'message' => $request->message,
                'data' => $request->data,
                'severity' => $request->severity,
                'context' => $request->context,
                'user_id' => auth()->check() ? auth()->id() : null,
            ]);

            return response()->json([
                'message' => 'Log registrado com sucesso',
                'data' => $log,
            ], 201);
        } catch (\Exception $e) {
            \Log::error('Erro ao salvar console log', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'message' => 'Erro ao salvar log',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Armazena múltiplos logs em batch (otimização de performance)
     */
    public function storeBatch(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'logs' => 'required|array|max:50', // Máximo de 50 logs por batch
            'logs.*.type' => 'required|string|max:50',
            'logs.*.message' => 'required|string',
            'logs.*.data' => 'nullable|string',
            'logs.*.severity' => 'required|string|in:error,warning,info',
            'logs.*.context' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Dados inválidos',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $userId = auth()->check() ? auth()->id() : null;
            $logs = [];

            foreach ($request->logs as $logData) {
                // Serializa context se for array (o campo é text, não json na migration)
                $context = null;
                if (isset($logData['context']) && $logData['context'] !== null) {
                    if (is_array($logData['context'])) {
                        $context = json_encode($logData['context'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                    } elseif (is_string($logData['context'])) {
                        $context = $logData['context'];
                    }
                }
                
                $logs[] = [
                    'type' => $logData['type'],
                    'message' => $logData['message'],
                    'data' => $logData['data'] ?? null,
                    'severity' => $logData['severity'],
                    'context' => $context,
                    'user_id' => $userId,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }

            // Insere em batch (muito mais rápido que múltiplos inserts)
            ConsoleLog::insert($logs);

            return response()->json([
                'message' => 'Logs registrados com sucesso',
                'count' => count($logs),
            ], 201);
        } catch (\Exception $e) {
            \Log::error('Erro ao salvar console logs em batch', [
                'error' => $e->getMessage(),
                'count' => count($request->logs ?? []),
            ]);

            return response()->json([
                'message' => 'Erro ao salvar logs',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Lista logs do console (apenas para administradores)
     */
    public function index(Request $request): JsonResponse
    {
        // Verifica se o usuário tem permissão (pode ser ajustado conforme necessário)
        if (!auth()->check() || auth()->user()->role !== 'admin') {
            return response()->json([
                'message' => 'Acesso negado',
            ], 403);
        }

        $query = ConsoleLog::with(['user', 'resolver'])
            ->orderBy('created_at', 'desc');

        // Filtros
        if ($request->has('type')) {
            $query->ofType($request->type);
        }

        if ($request->has('severity')) {
            $query->ofSeverity($request->severity);
        }

        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('recent')) {
            $hours = (int) $request->get('recent', 24);
            $query->recent($hours);
        }

        // Filtro de status (unique, repeated, obsolete, resolved, unresolved)
        $statusFilter = $request->get('status');
        if ($statusFilter === 'resolved') {
            $query->resolved();
        } elseif ($statusFilter === 'unresolved') {
            $query->unresolved();
        }

        // Exportação
        if ($request->has('export')) {
            $logs = $query->get();
            return $this->exportLogs($logs, $request->get('format', 'json'));
        }

        // Paginação
        $perPage = min((int) $request->get('per_page', 50), 100);
        $logs = $query->paginate($perPage);

        // Adiciona informações de repetição para cada log
        $logs->getCollection()->transform(function ($log) use ($query) {
            // Busca logs similares (mesma mensagem e tipo) nos últimos 7 dias
            $similarLogs = ConsoleLog::where('type', $log->type)
                ->where('message', $log->message)
                ->where('id', '!=', $log->id)
                ->where('created_at', '>=', now()->subDays(7))
                ->orderBy('created_at', 'asc')
                ->get();

            $log->occurrence_count = $similarLogs->count() + 1; // +1 para incluir o log atual
            $log->first_occurrence = $similarLogs->first()?->created_at ?? $log->created_at;
            $log->last_occurrence = $log->created_at;
            $log->is_repeated = $similarLogs->count() > 0;
            $log->is_obsolete = $log->created_at->lt(now()->subDays(7)); // Logs com mais de 7 dias são considerados obsoletos

            return $log;
        });

        // Aplica filtro de status após calcular repetições (exceto resolved/unresolved que já foi aplicado)
        $statusFilter = $request->get('status');
        if ($statusFilter && !in_array($statusFilter, ['resolved', 'unresolved'])) {
            $logs->setCollection($logs->getCollection()->filter(function ($log) use ($statusFilter) {
                return match($statusFilter) {
                    'unique' => !$log->is_repeated,
                    'repeated' => $log->is_repeated,
                    'obsolete' => $log->is_obsolete,
                    default => true,
                };
            }));
        }

        return response()->json($logs);
    }

    /**
     * Exporta logs em JSON ou CSV
     */
    private function exportLogs($logs, string $format = 'json')
    {
        if ($format === 'csv') {
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => 'attachment; filename="console-logs-' . date('Y-m-d-His') . '.csv"',
            ];

            $callback = function() use ($logs) {
                $file = fopen('php://output', 'w');
                
                // Cabeçalho
                fputcsv($file, ['ID', 'Data/Hora', 'Tipo', 'Severidade', 'Mensagem', 'Usuário', 'URL', 'User Agent']);
                
                // Dados
                foreach ($logs as $log) {
                    $context = $log->context ?? [];
                    fputcsv($file, [
                        $log->id,
                        $log->created_at->format('Y-m-d H:i:s'),
                        $log->type,
                        $log->severity,
                        $log->message,
                        $log->user ? $log->user->name : 'Não autenticado',
                        $context['url'] ?? '-',
                        $context['userAgent'] ?? '-',
                    ]);
                }
                
                fclose($file);
            };

            return response()->stream($callback, 200, $headers);
        }

        // JSON (padrão)
        return response()->json([
            'data' => $logs,
            'exported_at' => now()->toIso8601String(),
            'total' => $logs->count(),
        ], 200, [
            'Content-Type' => 'application/json',
            'Content-Disposition' => 'attachment; filename="console-logs-' . date('Y-m-d-His') . '.json"',
        ]);
    }

    /**
     * Mostra um log específico
     */
    public function show(ConsoleLog $consoleLog): JsonResponse
    {
        // Verifica permissão
        if (!auth()->check() || auth()->user()->role !== 'admin') {
            return response()->json([
                'message' => 'Acesso negado',
            ], 403);
        }

        $consoleLog->load('user');

        return response()->json([
            'data' => $consoleLog,
        ]);
    }

    /**
     * Marca um log como resolvido ou não resolvido
     */
    public function toggleResolved(Request $request, ConsoleLog $consoleLog): JsonResponse
    {
        // Verifica permissão
        if (!auth()->check() || auth()->user()->role !== 'admin') {
            return response()->json([
                'message' => 'Acesso negado',
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'is_resolved' => 'required|boolean',
            'resolution_notes' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Dados inválidos',
                'errors' => $validator->errors(),
            ], 422);
        }

        $consoleLog->is_resolved = $request->boolean('is_resolved');
        $consoleLog->resolved_at = $request->boolean('is_resolved') ? now() : null;
        $consoleLog->resolved_by = $request->boolean('is_resolved') ? auth()->id() : null;
        $consoleLog->resolution_notes = $request->input('resolution_notes');
        $consoleLog->save();

        $consoleLog->load('resolver');

        return response()->json([
            'message' => $consoleLog->is_resolved ? 'Log marcado como resolvido' : 'Log marcado como não resolvido',
            'data' => $consoleLog,
        ]);
    }

    /**
     * Deleta um log
     */
    public function destroy(ConsoleLog $consoleLog): JsonResponse
    {
        // Verifica permissão
        if (!auth()->check() || auth()->user()->role !== 'admin') {
            return response()->json([
                'message' => 'Acesso negado',
            ], 403);
        }

        $consoleLog->delete();

        return response()->json([
            'message' => 'Log deletado com sucesso',
        ], 204);
    }

    /**
     * Estatísticas de logs
     */
    public function stats(): JsonResponse
    {
        // Verifica permissão
        if (!auth()->check() || auth()->user()->role !== 'admin') {
            return response()->json([
                'message' => 'Acesso negado',
            ], 403);
        }

        $stats = [
            'total' => ConsoleLog::count(),
            'resolved' => ConsoleLog::resolved()->count(),
            'unresolved' => ConsoleLog::unresolved()->count(),
            'by_type' => ConsoleLog::selectRaw('type, count(*) as count')
                ->groupBy('type')
                ->pluck('count', 'type'),
            'by_severity' => ConsoleLog::selectRaw('severity, count(*) as count')
                ->groupBy('severity')
                ->pluck('count', 'severity'),
            'recent_errors' => ConsoleLog::ofSeverity('error')
                ->recent(24)
                ->count(),
            'recent_warnings' => ConsoleLog::ofSeverity('warning')
                ->recent(24)
                ->count(),
            'unresolved_errors' => ConsoleLog::ofSeverity('error')
                ->unresolved()
                ->count(),
        ];

        return response()->json([
            'data' => $stats,
        ]);
    }

    /**
     * Página web para visualizar logs
     */
    public function indexPage()
    {
        return view('console-logs.index');
    }
}
