<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Episode;
use App\Models\EpisodeAsset;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class EpisodeAssetController extends Controller
{
    /**
     * Lista assets de um episódio
     */
    public function index(Episode $episode)
    {
        $assets = $episode->assets()->orderBy('created_at')->get();

        return response()->json(['data' => $assets]);
    }

    /**
     * Cria um novo asset
     */
    public function store(Request $request, Episode $episode): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'asset_type' => ['required', 'string', 'in:audio,video,thumb,roteiro,cortes,drive,outros'],
            'label' => ['nullable', 'string', 'max:255'],
            'url' => ['required', 'url', 'max:500'],
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $asset = $episode->assets()->create($validator->validated());

        return response()->json(['data' => $asset], 201);
    }

    /**
     * Atualiza um asset
     */
    public function update(Request $request, Episode $episode, EpisodeAsset $asset): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'asset_type' => ['sometimes', 'string', 'in:audio,video,thumb,roteiro,cortes,drive,outros'],
            'label' => ['nullable', 'string', 'max:255'],
            'url' => ['sometimes', 'url', 'max:500'],
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $asset->update($validator->validated());

        return response()->json(['data' => $asset]);
    }

    /**
     * Remove um asset
     */
    public function destroy(Episode $episode, EpisodeAsset $asset): JsonResponse
    {
        $asset->delete();

        return response()->json(['message' => 'Asset deletado com sucesso'], 200);
    }
}
