<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreEpisodeRequest;
use App\Http\Requests\UpdateEpisodeRequest;
use App\Models\Episode;
use App\Models\Guest;
use App\Models\Theme;
use App\Services\EpisodeStatusService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class EpisodeController extends Controller
{
    public function __construct(
        private EpisodeStatusService $statusService
    ) {}

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Se for requisição JSON (API), retorna JSON
        if ($request->wantsJson() || $request->expectsJson()) {
            $query = Episode::with(['guests', 'themes', 'creator', 'tasks', 'assets']);

            // Filtros
            if ($request->has('status')) {
                $query->where('status', $request->status);
            }

            if ($request->has('theme_id')) {
                $query->whereHas('themes', function ($q) use ($request) {
                    $q->where('themes.id', $request->theme_id);
                });
            }

            if ($request->has('guest_id')) {
                $query->whereHas('guests', function ($q) use ($request) {
                    $q->where('guests.id', $request->guest_id);
                });
            }

            // Ordenação
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');
            $query->orderBy($sortBy, $sortOrder);

            // Paginação
            $perPage = $request->get('per_page', 15);
            $episodes = $query->paginate($perPage);

            return response()->json([
                'data' => $episodes->items(),
                'meta' => [
                    'current_page' => $episodes->currentPage(),
                    'from' => $episodes->firstItem(),
                    'last_page' => $episodes->lastPage(),
                    'per_page' => $episodes->perPage(),
                    'to' => $episodes->lastItem(),
                    'total' => $episodes->total(),
                ],
            ]);
        }

        // Retorna view para web
        $statuses = ['pauta', 'gravacao', 'edicao', 'revisao', 'agendado', 'publicado'];
        $themes = Theme::orderBy('name')->get();
        $guests = Guest::orderBy('name')->get();

        return view('episodes.index', compact('statuses', 'themes', 'guests'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $guests = Guest::orderBy('name')->get();
        $themes = Theme::orderBy('name')->get();
        
        return view('episodes.create', compact('guests', 'themes'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreEpisodeRequest $request)
    {
        $data = $request->validated();
        
        // Gerar slug se não fornecido
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['title']);
        }

        $data['created_by'] = auth()->id();

        $episode = Episode::create($data);

        // Sincronizar relacionamentos N:N
        if ($request->has('guests')) {
            $episode->guests()->sync($request->guests);
        }

        if ($request->has('themes')) {
            $episode->themes()->sync($request->themes);
        }

        $episode->load(['guests', 'themes', 'creator', 'tasks', 'assets']);

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['data' => $episode], 201);
        }

        return redirect()->route('episodes.show', $episode)
            ->with('success', 'Episódio criado com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Episode $episode)
    {
        $episode->load(['guests', 'themes', 'creator', 'tasks.assignee', 'assets', 'notes.creator', 'metrics']);

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['data' => $episode]);
        }

        return view('episodes.show', compact('episode'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Episode $episode)
    {
        $guests = Guest::orderBy('name')->get();
        $themes = Theme::orderBy('name')->get();
        $episode->load(['guests', 'themes']);
        
        return view('episodes.edit', compact('episode', 'guests', 'themes'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateEpisodeRequest $request, Episode $episode)
    {
        $data = $request->validated();

        // Gerar slug se título mudou e slug não foi fornecido
        if (isset($data['title']) && $episode->title !== $data['title'] && !isset($data['slug'])) {
            $data['slug'] = Str::slug($data['title']);
        }

        $episode->update($data);

        // Sincronizar relacionamentos N:N
        if ($request->has('guests')) {
            $episode->guests()->sync($request->guests);
        }

        if ($request->has('themes')) {
            $episode->themes()->sync($request->themes);
        }

        $episode->load(['guests', 'themes', 'creator', 'tasks', 'assets']);

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['data' => $episode]);
        }

        return redirect()->route('episodes.show', $episode)
            ->with('success', 'Episódio atualizado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Episode $episode)
    {
        $episode->delete();

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['message' => 'Episódio deletado com sucesso'], 200);
        }

        return redirect()->route('episodes.index')
            ->with('success', 'Episódio deletado com sucesso!');
    }

    /**
     * Avançar status do episódio
     */
    public function advanceStatus(Request $request, Episode $episode)
    {
        $to = $request->input('to') ?? $request->input('status');

        if (!$to) {
            return response()->json(['error' => 'Status de destino não informado'], 400);
        }

        $result = $this->statusService->transition($episode, $to);

        if (!$result['success']) {
            // Formata erros no padrão Laravel para o frontend
            return response()->json([
                'message' => 'Transição não permitida',
                'errors' => [
                    'status' => $result['errors'] ?? ['Não é possível realizar esta transição'],
                ],
            ], 422);
        }

        $episode->refresh();
        $episode->load(['guests', 'themes', 'creator', 'tasks', 'assets']);

        return response()->json([
            'message' => $result['message'],
            'data' => $episode,
        ]);
    }

    /**
     * Retorna transições permitidas para o status atual
     */
    public function allowedTransitions(Episode $episode)
    {
        $transitions = $this->statusService->getAllowedTransitions($episode->status);

        return response()->json([
            'data' => $transitions,
        ]);
    }
}
