<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Episode;
use App\Models\EpisodeTask;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class EpisodeTaskController extends Controller
{
    /**
     * Lista tarefas de um episódio
     */
    public function index(Episode $episode)
    {
        $tasks = $episode->tasks()->with('assignee')->orderBy('created_at')->get();

        if (request()->wantsJson() || request()->expectsJson()) {
            return response()->json(['data' => $tasks]);
        }

        return response()->json(['data' => $tasks]);
    }

    /**
     * Cria uma nova tarefa
     */
    public function store(Request $request, Episode $episode): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'title' => ['required', 'string', 'max:255'],
            'due_date' => ['nullable', 'date'],
            'assigned_to' => ['nullable', 'exists:users,id'],
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $task = $episode->tasks()->create($validator->validated());
        $task->load('assignee');

        return response()->json(['data' => $task], 201);
    }

    /**
     * Atualiza uma tarefa (incluindo toggle is_done)
     */
    public function update(Request $request, Episode $episode, EpisodeTask $task): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'title' => ['sometimes', 'string', 'max:255'],
            'is_done' => ['sometimes', 'boolean'],
            'due_date' => ['nullable', 'date'],
            'assigned_to' => ['nullable', 'exists:users,id'],
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $task->update($validator->validated());
        $task->load('assignee');

        return response()->json(['data' => $task]);
    }

    /**
     * Remove uma tarefa
     */
    public function destroy(Episode $episode, EpisodeTask $task): JsonResponse
    {
        $task->delete();

        return response()->json(['message' => 'Tarefa deletada com sucesso'], 200);
    }
}
