<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreGuestRequest;
use App\Models\Guest;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class GuestController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Se for requisição JSON (API), retorna JSON
        if ($request->wantsJson() || $request->expectsJson()) {
            $query = Guest::with('organization')->withCount('episodes');

            if ($request->has('search') || $request->has('q')) {
                $search = $request->get('search') ?? $request->get('q');
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('role_title', 'like', "%{$search}%")
                      ->orWhere('organization', 'like', "%{$search}%");
                });
            }

            $perPage = $request->get('per_page', 15);
            $guests = $query->orderBy('name')->paginate($perPage);

            return response()->json([
                'data' => $guests->items(),
                'meta' => [
                    'current_page' => $guests->currentPage(),
                    'from' => $guests->firstItem(),
                    'last_page' => $guests->lastPage(),
                    'per_page' => $guests->perPage(),
                    'to' => $guests->lastItem(),
                    'total' => $guests->total(),
                ],
            ]);
        }

        // Retorna view para web
        return view('guests.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('guests.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreGuestRequest $request)
    {
        $guest = Guest::create($request->validated());
        $guest->load('organization');

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['data' => $guest], 201);
        }

        return redirect()->route('guests.index')
            ->with('success', 'Convidado criado com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Guest $guest)
    {
        $guest->load(['organization', 'episodes.themes', 'episodes' => function ($q) {
            $q->orderBy('publish_date', 'desc');
        }]);

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['data' => $guest]);
        }

        return view('guests.show', compact('guest'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Guest $guest)
    {
        return view('guests.edit', compact('guest'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(StoreGuestRequest $request, Guest $guest)
    {
        $guest->update($request->validated());
        $guest->load('organization');

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['data' => $guest]);
        }

        return redirect()->route('guests.index')
            ->with('success', 'Convidado atualizado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Guest $guest)
    {
        $guest->delete();

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['message' => 'Convidado deletado com sucesso'], 200);
        }

        return redirect()->route('guests.index')
            ->with('success', 'Convidado deletado com sucesso!');
    }
}
