<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreProspectRequest;
use App\Http\Requests\UpdateProspectRequest;
use App\Models\Prospect;
use App\Models\Theme;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class ProspectController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View|JsonResponse
    {
        $query = Prospect::with(['creator', 'themes', 'guest']);

        // Filtros
        if ($request->has('status') && $request->status !== '') {
            $query->where('status', $request->status);
        }

        if ($request->has('origin') && $request->origin !== '') {
            $query->where('origin', $request->origin);
        }

        if ($request->has('type') && $request->type !== '') {
            $query->where('type', $request->type);
        }

        // Busca
        if ($request->has('search') && $request->search !== '') {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('company_name', 'like', "%{$search}%");
            });
        }

        // Ordenação
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Se for requisição JSON (API), retorna JSON
        if ($request->wantsJson() || $request->expectsJson()) {
            $perPage = $request->get('per_page', 15);
            $prospects = $query->paginate($perPage);

            return response()->json([
                'data' => $prospects->items(),
                'meta' => [
                    'current_page' => $prospects->currentPage(),
                    'from' => $prospects->firstItem(),
                    'last_page' => $prospects->lastPage(),
                    'per_page' => $prospects->perPage(),
                    'to' => $prospects->lastItem(),
                    'total' => $prospects->total(),
                ],
            ]);
        }

        // Retorna view para web
        $prospects = $query->paginate(15);
        $statuses = [
            Prospect::STATUS_TO_REVIEW => 'Para Revisar',
            Prospect::STATUS_APPROVED => 'Aprovado',
            Prospect::STATUS_INVITED => 'Convidado',
            Prospect::STATUS_IN_CONVERSATION => 'Em Conversa',
            Prospect::STATUS_SCHEDULED => 'Agendado',
            Prospect::STATUS_CONFIRMED => 'Confirmado',
            Prospect::STATUS_RECORDED => 'Gravado',
            Prospect::STATUS_PUBLISHED => 'Publicado',
            Prospect::STATUS_FOLLOW_UP => 'Follow-up',
            Prospect::STATUS_REJECTED => 'Rejeitado',
            Prospect::STATUS_ARCHIVED => 'Arquivado',
        ];

        $origins = [
            Prospect::ORIGIN_INBOUND_FORM => 'Formulário Público',
            Prospect::ORIGIN_OUTBOUND_CURATED => 'Curadoria Externa',
            Prospect::ORIGIN_REFERRAL => 'Indicação',
            Prospect::ORIGIN_EVENT => 'Evento',
            Prospect::ORIGIN_OTHER => 'Outro',
        ];

        return view('prospects.index', compact('prospects', 'statuses', 'origins'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        $themes = Theme::orderBy('name')->get();
        return view('prospects.create', compact('themes'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreProspectRequest $request): RedirectResponse
    {
        $prospect = Prospect::create([
            'type' => $request->type,
            'name' => $request->name,
            'email' => $request->email,
            'whatsapp' => $request->whatsapp,
            'linkedin_url' => $request->linkedin_url,
            'website_url' => $request->website_url,
            'role_title' => $request->role_title,
            'company_name' => $request->company_name,
            'organization_type' => $request->organization_type,
            'sector' => $request->sector,
            'location' => $request->location,
            'what_they_do' => $request->what_they_do,
            'problem_solved' => $request->problem_solved,
            'stage' => $request->stage,
            'expertise_topics' => $request->expertise_topics,
            'story_case' => $request->story_case,
            'ecosystem_request' => $request->ecosystem_request,
            'preferred_format' => $request->preferred_format,
            'preferred_time' => $request->preferred_time,
            'availability_window' => $request->availability_window,
            'consent_given' => $request->has('consent_given'),
            'origin' => $request->origin ?? Prospect::ORIGIN_OUTBOUND_CURATED,
            'status' => $request->status ?? Prospect::STATUS_TO_REVIEW,
            'priority' => $request->priority,
            'notes' => $request->notes,
            'fit_score' => $request->fit_score,
            'impact_potential' => $request->impact_potential,
            'timing_score' => $request->timing_score,
            'diversity_flags' => $request->diversity_flags,
            'preferred_slots' => $request->preferred_slots,
            'scheduled_at' => $request->scheduled_at,
            'meeting_link' => $request->meeting_link,
            'created_by' => auth()->id(),
        ]);

        // Associar temas
        if ($request->has('themes')) {
            $prospect->themes()->attach($request->themes);
        }

        return redirect()->route('prospects.show', $prospect)
            ->with('success', 'Prospect criado com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Prospect $prospect): View
    {
        $prospect->load(['creator', 'themes', 'guest']);
        return view('prospects.show', compact('prospect'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Prospect $prospect): View
    {
        $themes = Theme::orderBy('name')->get();
        $prospect->load('themes');
        return view('prospects.edit', compact('prospect', 'themes'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateProspectRequest $request, Prospect $prospect): RedirectResponse
    {
        $data = $request->validated();
        
        // Tratar consent_given como boolean
        if ($request->has('consent_given')) {
            $data['consent_given'] = $request->boolean('consent_given');
        }
        
        $prospect->update($data);

        // Atualizar temas
        if ($request->has('themes')) {
            $prospect->themes()->sync($request->themes);
        } else {
            $prospect->themes()->detach();
        }

        return redirect()->route('prospects.show', $prospect)
            ->with('success', 'Prospect atualizado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Prospect $prospect): RedirectResponse
    {
        $prospect->delete();

        return redirect()->route('prospects.index')
            ->with('success', 'Prospect removido com sucesso!');
    }

    /**
     * Aprovar prospect
     */
    public function approve(Prospect $prospect): RedirectResponse
    {
        $prospect->update([
            'status' => Prospect::STATUS_APPROVED,
        ]);

        return redirect()->back()
            ->with('success', 'Prospect aprovado com sucesso!');
    }

    /**
     * Rejeitar prospect
     */
    public function reject(Request $request, Prospect $prospect): RedirectResponse
    {
        $request->validate([
            'rejection_reason' => 'required|string|max:500',
        ]);

        $prospect->update([
            'status' => Prospect::STATUS_REJECTED,
            'rejection_reason' => $request->rejection_reason,
        ]);

        return redirect()->back()
            ->with('success', 'Prospect rejeitado com sucesso!');
    }
}
