<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreThemeRequest;
use App\Models\Theme;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ThemeController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Se for requisição JSON (API), retorna JSON
        if ($request->wantsJson() || $request->expectsJson()) {
            $query = Theme::withCount('episodes');

            if ($request->has('search') || $request->has('q')) {
                $search = $request->get('search') ?? $request->get('q');
                $query->where('name', 'like', "%{$search}%");
            }

            $themes = $query->orderBy('name')->get();

            return response()->json(['data' => $themes]);
        }

        // Retorna view para web
        return view('themes.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('themes.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreThemeRequest $request)
    {
        $theme = Theme::create($request->validated());

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['data' => $theme], 201);
        }

        return redirect()->route('themes.index')
            ->with('success', 'Tema criado com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Theme $theme)
    {
        $theme->load(['episodes' => function ($q) {
            $q->orderBy('publish_date', 'desc');
        }]);

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['data' => $theme]);
        }

        return view('themes.show', compact('theme'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Theme $theme)
    {
        return view('themes.edit', compact('theme'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(StoreThemeRequest $request, Theme $theme)
    {
        $theme->update($request->validated());

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['data' => $theme]);
        }

        return redirect()->route('themes.index')
            ->with('success', 'Tema atualizado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Theme $theme)
    {
        $theme->delete();

        if ($request->wantsJson() || $request->expectsJson()) {
            return response()->json(['message' => 'Tema deletado com sucesso'], 200);
        }

        return redirect()->route('themes.index')
            ->with('success', 'Tema deletado com sucesso!');
    }
}
