<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Episode extends Model
{
    use HasFactory;
    protected $fillable = [
        'title',
        'slug',
        'record_date',
        'publish_date',
        'status',
        'description',
        'tags',
        'youtube_url',
        'spotify_url',
        'created_by',
    ];

    protected $casts = [
        'record_date' => 'date',
        'publish_date' => 'date',
    ];

    /**
     * Usuário que criou o episódio
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Convidados do episódio
     */
    public function guests(): BelongsToMany
    {
        return $this->belongsToMany(Guest::class, 'episode_guests')
            ->withPivot('guest_role')
            ->withTimestamps();
    }

    /**
     * Temas abordados no episódio
     */
    public function themes(): BelongsToMany
    {
        return $this->belongsToMany(Theme::class, 'episode_themes')
            ->withTimestamps();
    }

    /**
     * Tarefas do episódio
     */
    public function tasks(): HasMany
    {
        return $this->hasMany(EpisodeTask::class);
    }

    /**
     * Assets do episódio
     */
    public function assets(): HasMany
    {
        return $this->hasMany(EpisodeAsset::class);
    }

    /**
     * Notas/comentários do episódio
     */
    public function notes(): HasMany
    {
        return $this->hasMany(EpisodeNote::class);
    }

    /**
     * Métricas do episódio
     */
    public function metrics(): HasMany
    {
        return $this->hasMany(EpisodeMetric::class);
    }
}
