<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Prospect extends Model
{
    use HasFactory;

    protected $fillable = [
        // Identificação
        'type',
        'name',
        'email',
        'whatsapp',
        'linkedin_url',
        'website_url',
        
        // Perfil
        'role_title',
        'company_name',
        'organization_type',
        'sector',
        'location',
        
        // Sobre a Iniciativa
        'what_they_do',
        'problem_solved',
        'stage',
        
        // Pauta e Valor
        'expertise_topics',
        'story_case',
        'ecosystem_request',
        
        // Disponibilidade
        'preferred_format',
        'preferred_time',
        'availability_window',
        'consent_given',
        
        // Curadoria
        'origin',
        'status',
        'priority',
        'notes',
        'fit_score',
        'impact_potential',
        'timing_score',
        'total_score',
        'diversity_flags',
        'rejection_reason',
        
        // Agendamento
        'preferred_slots',
        'scheduled_at',
        'meeting_link',
        
        'created_by',
    ];

    protected $casts = [
        'diversity_flags' => 'array',
        'priority' => 'integer',
        'fit_score' => 'integer',
        'impact_potential' => 'integer',
        'timing_score' => 'integer',
        'total_score' => 'integer',
        'consent_given' => 'boolean',
        'scheduled_at' => 'datetime',
    ];

    // Status disponíveis
    const STATUS_TO_REVIEW = 'to_review';
    const STATUS_APPROVED = 'approved';
    const STATUS_INVITED = 'invited';
    const STATUS_CONTACTED = 'contacted';
    const STATUS_IN_CONVERSATION = 'in_conversation';
    const STATUS_SCHEDULED = 'scheduled';
    const STATUS_CONFIRMED = 'confirmed';
    const STATUS_RECORDED = 'recorded';
    const STATUS_PUBLISHED = 'published';
    const STATUS_FOLLOW_UP = 'follow_up';
    const STATUS_REJECTED = 'rejected';
    const STATUS_ARCHIVED = 'archived';
    
    // Origens
    const ORIGIN_INBOUND_FORM = 'inbound_form';
    const ORIGIN_OUTBOUND_CURATED = 'outbound_curated';
    const ORIGIN_REFERRAL = 'referral';
    const ORIGIN_EVENT = 'event';
    const ORIGIN_OTHER = 'other';

    // Tipos
    const TYPE_PERSON = 'person';
    const TYPE_COMPANY = 'company';

    /**
     * Usuário que criou o prospect
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Temas relacionados ao prospect
     */
    public function themes(): BelongsToMany
    {
        return $this->belongsToMany(Theme::class, 'prospect_themes')
            ->withTimestamps();
    }

    /**
     * Guest convertido a partir deste prospect
     */
    public function guest(): \Illuminate\Database\Eloquent\Relations\HasOne
    {
        return $this->hasOne(Guest::class, 'prospect_id');
    }

    /**
     * Calcula e atualiza o score total
     * Recalculado automaticamente no saving (via booted)
     */
    public function calculateTotalScore(): void
    {
        $this->total_score = ($this->fit_score ?? 0) 
            + ($this->impact_potential ?? 0) 
            + ($this->timing_score ?? 0);
    }
    
    /**
     * Recalcula score total antes de salvar
     */
    protected static function booted(): void
    {
        static::saving(function (Prospect $prospect) {
            $prospect->calculateTotalScore();
        });
    }
}
