<?php

namespace App\Services;

use App\Models\Episode;
use App\Models\EpisodeTask;
use App\DataTransferObjects\CalendarEvent;
use Illuminate\Support\Collection;
use Carbon\Carbon;

class CalendarService
{
    /**
     * Busca eventos do calendário para um período
     * 
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @param array $filters ['event_types' => [], 'statuses' => []]
     * @return Collection<CalendarEvent>
     */
    public function getEvents(Carbon $startDate, Carbon $endDate, array $filters = []): Collection
    {
        $events = collect();
        
        // Eventos de gravação
        if (empty($filters['event_types']) || in_array('recording', $filters['event_types'])) {
            $events = $events->merge($this->getRecordingEvents($startDate, $endDate, $filters));
        }
        
        // Eventos de publicação
        if (empty($filters['event_types']) || in_array('publication', $filters['event_types'])) {
            $events = $events->merge($this->getPublicationEvents($startDate, $endDate, $filters));
        }
        
        // Eventos de tarefas
        if (empty($filters['event_types']) || in_array('task', $filters['event_types'])) {
            $events = $events->merge($this->getTaskEvents($startDate, $endDate, $filters));
        }
        
        return $events->sortBy('date');
    }
    
    /**
     * Eventos de gravação (record_date)
     */
    private function getRecordingEvents(Carbon $start, Carbon $end, array $filters): Collection
    {
        $query = Episode::whereNotNull('record_date')
            ->whereBetween('record_date', [$start, $end])
            ->with(['guests', 'themes']);
            
        if (!empty($filters['statuses'])) {
            $query->whereIn('status', $filters['statuses']);
        }
        
        return $query->get()->map(function ($episode) {
            return new CalendarEvent(
                id: "recording-{$episode->id}",
                type: 'recording',
                date: $episode->record_date,
                title: "Gravação: {$episode->title}",
                episode_id: $episode->id,
                episode: $episode,
                color: 'blue',
                icon: 'mic',
            );
        });
    }
    
    /**
     * Eventos de publicação (publish_date)
     */
    private function getPublicationEvents(Carbon $start, Carbon $end, array $filters): Collection
    {
        $query = Episode::whereNotNull('publish_date')
            ->whereBetween('publish_date', [$start->format('Y-m-d'), $end->format('Y-m-d')])
            ->with(['guests', 'themes']);
            
        if (!empty($filters['statuses'])) {
            $query->whereIn('status', $filters['statuses']);
        }
        
        return $query->get()->map(function ($episode) {
            return new CalendarEvent(
                id: "publication-{$episode->id}",
                type: 'publication',
                date: $episode->publish_date,
                title: "Publicação: {$episode->title}",
                episode_id: $episode->id,
                episode: $episode,
                color: 'green',
                icon: 'broadcast',
            );
        });
    }
    
    /**
     * Eventos de tarefas (due_date)
     */
    private function getTaskEvents(Carbon $start, Carbon $end, array $filters): Collection
    {
        $query = EpisodeTask::whereNotNull('due_date')
            ->whereBetween('due_date', [$start, $end])
            ->with(['episode', 'assignee']);
            
        if (!empty($filters['statuses'])) {
            $query->whereHas('episode', function ($q) use ($filters) {
                $q->whereIn('status', $filters['statuses']);
            });
        }
        
        return $query->get()->map(function ($task) {
            return new CalendarEvent(
                id: "task-{$task->id}",
                type: 'task',
                date: $task->due_date,
                title: $task->title,
                episode_id: $task->episode_id,
                episode: $task->episode,
                task: $task,
                color: $task->is_done ? 'gray' : 'orange',
                icon: 'check-circle',
                is_done: $task->is_done,
            );
        });
    }
    
    /**
     * Eventos agrupados por mês (para visão mensal)
     */
    public function getEventsByMonth(int $year, int $month, array $filters = []): Collection
    {
        $start = Carbon::create($year, $month, 1)->startOfMonth();
        $end = $start->copy()->endOfMonth();
        
        $events = $this->getEvents($start, $end, $filters);
        
        return $events->groupBy(function ($event) {
            return $event->date->format('Y-m-d');
        });
    }
}
