<?php

namespace App\Services;

use App\Models\Episode;
use Illuminate\Support\Collection;

class EpisodeStatusService
{
    // Transições permitidas
    private const ALLOWED_TRANSITIONS = [
        'pauta' => ['gravacao'],
        'gravacao' => ['edicao'],
        'edicao' => ['revisao'],
        'revisao' => ['agendado'],
        'agendado' => ['publicado'],
    ];
    
    // Regras de validação por transição
    private const VALIDATION_RULES = [
        'agendado' => [
            'requires' => ['youtube_url', 'spotify_url'], // pelo menos um
            'critical_tasks_done' => true,
        ],
    ];

    /**
     * Verifica se uma transição de status é permitida
     */
    public function canTransition(string $from, string $to): bool
    {
        if (!isset(self::ALLOWED_TRANSITIONS[$from])) {
            return false;
        }

        return in_array($to, self::ALLOWED_TRANSITIONS[$from]);
    }

    /**
     * Valida se a transição pode ser executada
     */
    public function validateTransition(Episode $episode, string $to): Collection
    {
        $errors = collect();
        
        $from = $episode->status;
        
        // Verifica se a transição é permitida
        if (!$this->canTransition($from, $to)) {
            $errors->push("Transição de '{$from}' para '{$to}' não é permitida.");
            return $errors;
        }

        // Aplica regras específicas de validação
        if (isset(self::VALIDATION_RULES[$to])) {
            $rules = self::VALIDATION_RULES[$to];
            
            // Validação de URLs (pelo menos uma deve estar preenchida)
            if (isset($rules['requires'])) {
                $hasRequired = false;
                foreach ($rules['requires'] as $field) {
                    if (!empty($episode->$field)) {
                        $hasRequired = true;
                        break;
                    }
                }
                
                if (!$hasRequired) {
                    $errors->push("Não é possível publicar um episódio sem links de publicação (YouTube ou Spotify).");
                }
            }
            
            // Validação de tarefas críticas
            if (isset($rules['critical_tasks_done']) && $rules['critical_tasks_done']) {
                // Busca tarefas críticas do template
                $defaultTasks = config('episode_tasks.default_tasks', []);
                $criticalTaskTitles = collect($defaultTasks)
                    ->where('is_critical', true)
                    ->pluck('title')
                    ->toArray();
                
                // Verifica se todas as tarefas críticas estão concluídas
                $unfinishedCriticalTasks = $episode->tasks()
                    ->whereIn('title', $criticalTaskTitles)
                    ->where('is_done', false)
                    ->count();
                
                if ($unfinishedCriticalTasks > 0) {
                    $errors->push("Existem tarefas obrigatórias pendentes no checklist.");
                }
            }
        }

        return $errors;
    }

    /**
     * Executa a transição de status com validação
     */
    public function transition(Episode $episode, string $to): array
    {
        $errors = $this->validateTransition($episode, $to);
        
        if ($errors->isNotEmpty()) {
            return [
                'success' => false,
                'errors' => $errors->toArray(),
            ];
        }

        $episode->status = $to;
        $episode->save();

        // Log da transição (pode ser adicionado em episode_notes ou tabela específica)
        $episode->notes()->create([
            'note' => "Status alterado de '{$episode->getOriginal('status')}' para '{$to}' por " . auth()->user()->name,
            'created_by' => auth()->id(),
        ]);

        return [
            'success' => true,
            'message' => "Status alterado para '{$to}' com sucesso.",
        ];
    }

    /**
     * Retorna as transições permitidas para um status
     */
    public function getAllowedTransitions(string $status): array
    {
        return self::ALLOWED_TRANSITIONS[$status] ?? [];
    }
}
