/**
 * Helper para chamadas de API
 * Inclui CSRF token automaticamente e trata erros
 */

/**
 * Obtém o token CSRF da meta tag
 */
function getCsrfToken() {
    const meta = document.querySelector('meta[name="csrf-token"]');
    return meta ? meta.getAttribute('content') : '';
}

/**
 * Contador de erros consecutivos para banner de API indisponível
 */
let apiErrorCount = 0;
const MAX_ERRORS_FOR_BANNER = 3;

/**
 * Exibe banner global de erro de API
 */
function showApiErrorBanner() {
    // Remove banner existente se houver
    const existingBanner = document.getElementById('api-error-banner');
    if (existingBanner) {
        existingBanner.remove();
    }

    // Cria novo banner
    const banner = document.createElement('div');
    banner.id = 'api-error-banner';
    banner.className = 'fixed top-0 left-0 right-0 z-50 bg-red-600 text-white p-4 shadow-lg';
    banner.innerHTML = `
        <div class="max-w-7xl mx-auto flex items-center justify-between">
            <div class="flex items-center gap-3">
                <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
                </svg>
                <span class="font-medium">API indisponível. Verifique sua conexão.</span>
            </div>
            <button 
                onclick="this.closest('#api-error-banner').remove(); window.location.reload();" 
                class="bg-red-700 hover:bg-red-800 px-4 py-2 rounded font-medium transition-colors"
            >
                Tentar novamente
            </button>
        </div>
    `;
    document.body.prepend(banner);
}

/**
 * Remove banner de erro de API
 */
function hideApiErrorBanner() {
    const banner = document.getElementById('api-error-banner');
    if (banner) {
        banner.remove();
    }
    apiErrorCount = 0;
}

/**
 * Trata erros HTTP de forma padronizada
 */
async function handleError(response) {
    const contentType = response.headers.get('content-type');
    const isJson = contentType && contentType.includes('application/json');
    
    let errorData = null;
    if (isJson) {
        errorData = await response.json();
    }

    // Incrementa contador de erros
    apiErrorCount++;

    // Trata diferentes códigos de erro
    switch (response.status) {
        case 401:
        case 419:
            // Sessão expirada ou não autenticado
            hideApiErrorBanner();
            window.location.href = '/login';
            throw new Error('Sessão expirada. Redirecionando para login...');

        case 403:
            // Sem permissão
            if (window.showToast) {
                window.showToast('Você não tem permissão para esta ação', 'error');
            }
            throw new Error('Acesso negado');

        case 422:
            // Erro de validação ou regra de negócio
            return {
                errors: errorData?.errors || {},
                message: errorData?.message || 'Erro de validação',
            };

        case 500:
        case 503:
            // Erro do servidor
            if (apiErrorCount >= MAX_ERRORS_FOR_BANNER) {
                showApiErrorBanner();
            }
            throw new Error(errorData?.message || 'Erro interno do servidor');

        default:
            // Outros erros
            if (apiErrorCount >= MAX_ERRORS_FOR_BANNER) {
                showApiErrorBanner();
            }
            throw new Error(errorData?.message || `Erro ${response.status}: ${response.statusText}`);
    }
}

/**
 * Helper principal para chamadas de API
 * 
 * @param {string} url - URL da API
 * @param {object} options - Opções do fetch (method, body, headers, etc.)
 * @returns {Promise<{data: any}>} - Dados da resposta
 */
export async function apiFetch(url, options = {}) {
    // Reset contador em caso de sucesso
    apiErrorCount = 0;
    hideApiErrorBanner();

    // Configura headers padrão
    const headers = {
        'Accept': 'application/json',
        'Content-Type': 'application/json',
        'X-Requested-With': 'XMLHttpRequest',
        ...options.headers,
    };

    // Adiciona CSRF token para métodos mutáveis
    const method = (options.method || 'GET').toUpperCase();
    if (['POST', 'PUT', 'PATCH', 'DELETE'].includes(method)) {
        headers['X-CSRF-TOKEN'] = getCsrfToken();
    }

    // Prepara body
    let body = options.body;
    if (body && typeof body === 'object' && !(body instanceof FormData)) {
        body = JSON.stringify(body);
    }

    try {
        const response = await fetch(url, {
            ...options,
            method,
            headers,
            body,
        });

        // Sucesso
        if (response.ok) {
            // 204 No Content não tem body
            if (response.status === 204) {
                return { data: null };
            }

            const contentType = response.headers.get('content-type');
            if (contentType && contentType.includes('application/json')) {
                const json = await response.json();
                // Se tiver 'data' e 'meta', preserva ambos (paginação)
                if (json.data !== undefined && json.meta !== undefined) {
                    return { data: json.data, meta: json.meta };
                }
                // Se tiver apenas 'data', retorna apenas o data
                if (json.data !== undefined) {
                    return { data: json.data };
                }
                // Caso contrário, retorna o JSON completo
                return { data: json };
            }

            return { data: await response.text() };
        }

        // Erro
        return await handleError(response);

    } catch (error) {
        // Erro de rede ou outro erro
        apiErrorCount++;
        if (apiErrorCount >= MAX_ERRORS_FOR_BANNER) {
            showApiErrorBanner();
        }
        throw error;
    }
}

// Exporta funções auxiliares
export { getCsrfToken, showApiErrorBanner, hideApiErrorBanner };
