/**
 * Componente Alpine.js para o Calendário Editorial
 */
export function calendarEditorial() {
    return {
        currentYear: new Date().getFullYear(),
        currentMonth: new Date().getMonth() + 1,
        events: {},
        loading: true,
        showEventModal: false,
        selectedEvent: null,
        weekDays: ['Dom', 'Seg', 'Ter', 'Qua', 'Qui', 'Sex', 'Sáb'],
        filters: {
            event_types: ['recording', 'publication', 'task'],
            statuses: [],
        },
        
        get currentMonthLabel() {
            const date = new Date(this.currentYear, this.currentMonth - 1, 1);
            return date.toLocaleDateString('pt-BR', { month: 'long', year: 'numeric' });
        },
        
        get calendarWeeks() {
            const firstDay = new Date(this.currentYear, this.currentMonth - 1, 1);
            const lastDay = new Date(this.currentYear, this.currentMonth, 0);
            const daysInMonth = lastDay.getDate();
            const startDayOfWeek = firstDay.getDay();
            
            const weeks = [];
            let currentWeek = [];
            
            // Preencher dias do mês anterior
            const prevMonth = new Date(this.currentYear, this.currentMonth - 1, 0);
            const daysInPrevMonth = prevMonth.getDate();
            for (let i = startDayOfWeek - 1; i >= 0; i--) {
                const day = daysInPrevMonth - i;
                currentWeek.push({
                    day: day,
                    date: `${this.currentYear}-${String(this.currentMonth - 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`,
                    isCurrentMonth: false,
                    isToday: false,
                });
            }
            
            // Dias do mês atual
            for (let day = 1; day <= daysInMonth; day++) {
                const date = `${this.currentYear}-${String(this.currentMonth).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
                const today = new Date();
                const isToday = today.getFullYear() === this.currentYear && 
                                today.getMonth() + 1 === this.currentMonth && 
                                today.getDate() === day;
                
                currentWeek.push({
                    day: day,
                    date: date,
                    isCurrentMonth: true,
                    isToday: isToday,
                });
                
                if (currentWeek.length === 7) {
                    weeks.push(currentWeek);
                    currentWeek = [];
                }
            }
            
            // Preencher dias do próximo mês
            const remainingDays = 7 - currentWeek.length;
            for (let day = 1; day <= remainingDays; day++) {
                const nextMonth = this.currentMonth === 12 ? 1 : this.currentMonth + 1;
                const nextYear = this.currentMonth === 12 ? this.currentYear + 1 : this.currentYear;
                currentWeek.push({
                    day: day,
                    date: `${nextYear}-${String(nextMonth).padStart(2, '0')}-${String(day).padStart(2, '0')}`,
                    isCurrentMonth: false,
                    isToday: false,
                });
            }
            
            if (currentWeek.length > 0) {
                weeks.push(currentWeek);
            }
            
            return weeks;
        },
        
        init() {
            this.loadEvents();
        },
        
        async loadEvents() {
            this.loading = true;
            try {
                const params = new URLSearchParams({
                    year: this.currentYear,
                    month: this.currentMonth,
                    view: 'month',
                });
                
                // Adicionar filtros
                if (this.filters.event_types.length > 0) {
                    this.filters.event_types.forEach(type => {
                        params.append('event_types[]', type);
                    });
                }
                
                if (this.filters.statuses.length > 0) {
                    this.filters.statuses.forEach(status => {
                        params.append('statuses[]', status);
                    });
                }
                
                const response = await window.apiFetch(`/api/calendar?${params.toString()}`);
                
                // A resposta vem como { data: { events: { '2026-01-15': [...], ... } } }
                this.events = response.data?.events || {};
            } catch (error) {
                console.error('Erro ao carregar eventos:', error);
                window.showToast('Erro ao carregar calendário', 'error');
            } finally {
                this.loading = false;
            }
        },
        
        getEventsForDay(date) {
            const dateStr = date.split('T')[0]; // Remove hora se houver
            return this.events[dateStr] || [];
        },
        
        previousMonth() {
            if (this.currentMonth === 1) {
                this.currentMonth = 12;
                this.currentYear--;
            } else {
                this.currentMonth--;
            }
            this.loadEvents();
        },
        
        nextMonth() {
            if (this.currentMonth === 12) {
                this.currentMonth = 1;
                this.currentYear++;
            } else {
                this.currentMonth++;
            }
            this.loadEvents();
        },
        
        goToToday() {
            const today = new Date();
            this.currentYear = today.getFullYear();
            this.currentMonth = today.getMonth() + 1;
            this.loadEvents();
        },
        
        clearFilters() {
            this.filters = {
                event_types: ['recording', 'publication', 'task'],
                statuses: [],
            };
            this.loadEvents();
        },
        
        openEventModal(event) {
            this.selectedEvent = event;
            this.showEventModal = true;
        },
        
        openDayModal(date) {
            // TODO: Implementar modal com todos os eventos do dia
            const events = this.getEventsForDay(date);
            if (events.length > 0) {
                this.openEventModal(events[0]);
            }
        },
        
        getEventColorClass(event) {
            const colorMap = {
                'blue': 'bg-blue-100 text-blue-800 hover:bg-blue-200',
                'green': 'bg-green-100 text-green-800 hover:bg-green-200',
                'orange': 'bg-orange-100 text-orange-800 hover:bg-orange-200',
                'gray': 'bg-gray-100 text-gray-800 hover:bg-gray-200',
            };
            return colorMap[event.color] || colorMap['gray'];
        },
        
        getEventIcon(event) {
            const iconMap = {
                'mic': '🎙',
                'broadcast': '📢',
                'check-circle': '📝',
            };
            return iconMap[event.icon] || '●';
        },
        
        formatDate(dateStr) {
            if (!dateStr) return '';
            const date = new Date(dateStr);
            return date.toLocaleDateString('pt-BR');
        },
        
        getBadgeStatus(status) {
            const statusMap = {
                'pauta': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">Pauta</span>',
                'gravacao': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">Gravação</span>',
                'edicao': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">Edição</span>',
                'revisao': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">Revisão</span>',
                'agendado': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">Agendado</span>',
                'publicado': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">Publicado</span>',
            };
            return statusMap[status] || statusMap['pauta'];
        }
    }
}
