/**
 * Sistema de rastreamento de erros do console
 * Captura console.error, console.warn, erros não tratados e promessas rejeitadas
 */

// Buffer para armazenar logs temporariamente
let logBuffer = [];
let bufferSize = 20; // Aumentado para reduzir requisições
let flushInterval = 60000; // 60 segundos (aumentado)
let isInitialized = false;
let flushTimer = null;
let debugMode = false;
let isEnabled = true; // Flag para habilitar/desabilitar

// Rate limiting: máximo de logs por minuto
let maxLogsPerMinute = 30;
let logsSentThisMinute = 0;
let rateLimitResetTime = Date.now() + 60000;

// Limite de logs por sessão (evita acúmulo excessivo)
let maxLogsPerSession = 200;
let totalLogsInSession = 0;

// Debounce para evitar envios muito frequentes
let debounceTimer = null;
let debounceDelay = 5000; // 5 segundos

// Padrões de URLs/strings para ignorar (extensões do navegador, serviços externos)
const IGNORE_PATTERNS = [
    /chrome-extension:\/\//i,
    /moz-extension:\/\//i,
    /safari-extension:\/\//i,
    /edge-extension:\/\//i,
    /^chrome:\/\//i,
    /^moz:\/\//i,
    /^safari:\/\//i,
    /^edge:\/\//i,
    /tactiq\.io/i,
    /extension/i,
    /content-scripts/i,
    /^Fetch finished loading:/i,
    /^XHR finished loading:/i,
    /DevTools/i,
    /^DevTools/i,
];

// URLs permitidas (domínio da aplicação)
let allowedOrigins = [];

/**
 * Coleta informações do contexto do erro
 */
function collectContext() {
    return {
        url: window.location.href,
        userAgent: navigator.userAgent,
        timestamp: new Date().toISOString(),
        viewport: {
            width: window.innerWidth,
            height: window.innerHeight,
        },
        referrer: document.referrer || null,
    };
}

/**
 * Captura stack trace de forma segura
 */
function getStackTrace(error) {
    if (error && error.stack) {
        return error.stack;
    }
    try {
        throw new Error();
    } catch (e) {
        return e.stack || 'Stack trace não disponível';
    }
}

/**
 * Verifica rate limiting
 */
function checkRateLimit() {
    const now = Date.now();
    
    // Reset contador a cada minuto
    if (now >= rateLimitResetTime) {
        logsSentThisMinute = 0;
        rateLimitResetTime = now + 60000;
    }
    
    // Se excedeu o limite, retorna false
    if (logsSentThisMinute >= maxLogsPerMinute) {
        if (debugMode) {
            console.warn('[Console Tracker] Rate limit atingido. Aguardando...');
        }
        return false;
    }
    
    return true;
}

/**
 * Envia logs em batch para o backend (otimização de performance)
 */
async function sendLogsBatch(logs) {
    if (!isEnabled || logs.length === 0) return;
    
    // Verifica rate limiting
    if (!checkRateLimit()) {
        // Se excedeu o limite, mantém os logs no buffer para tentar depois
        logBuffer.push(...logs);
        return;
    }
    
    try {
        if (debugMode) {
            console.log('[Console Tracker] Enviando batch de', logs.length, 'logs');
        }

        // Envia todos os logs em uma única requisição
        const response = await fetch('/api/console-logs/batch', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': getCsrfToken(),
                'X-Requested-With': 'XMLHttpRequest',
            },
            body: JSON.stringify({ logs }),
        });

        if (!response.ok) {
            const errorText = await response.text();
            if (debugMode) {
                console.warn('[Console Tracker] Falha ao enviar batch:', response.status, errorText);
            }
            // Em caso de erro, mantém os logs no buffer (serão tentados novamente no próximo flush)
            logBuffer.push(...logs);
        } else {
            logsSentThisMinute += logs.length;
            totalLogsInSession += logs.length;
            
            if (debugMode) {
                const result = await response.json();
                console.log('[Console Tracker] Batch enviado com sucesso:', result);
            }
        }
    } catch (error) {
        // Em caso de erro de rede, mantém os logs no buffer
        logBuffer.push(...logs);
        if (debugMode) {
            console.error('[Console Tracker] Erro ao enviar batch:', error);
        }
    }
}

/**
 * Obtém o token CSRF
 */
function getCsrfToken() {
    const meta = document.querySelector('meta[name="csrf-token"]');
    return meta ? meta.getAttribute('content') : '';
}

/**
 * Verifica se um log deve ser ignorado
 */
function shouldIgnoreLog(message, stack = null) {
    // Ignora logs vazios ou muito curtos
    if (!message || message.trim().length < 3) {
        return true;
    }

    // Verifica padrões de ignorar na mensagem
    for (const pattern of IGNORE_PATTERNS) {
        if (pattern.test(message)) {
            return true;
        }
    }

    // Verifica padrões no stack trace
    if (stack) {
        for (const pattern of IGNORE_PATTERNS) {
            if (pattern.test(stack)) {
                return true;
            }
        }
    }

    // Verifica se a mensagem parece ser de extensão (contém IDs de extensão no início)
    // Mas não bloqueia mensagens que apenas mencionam "extension" no meio
    if (/^[a-f0-9]{32}/i.test(message)) {
        return true;
    }
    
    // Só bloqueia se "extension" aparecer em contexto de URL ou caminho
    if (/extension:\/\/|content-scripts|extension\//i.test(message)) {
        return true;
    }

    return false;
}

/**
 * Verifica se um erro não tratado deve ser ignorado
 */
function shouldIgnoreUnhandledError(event) {
    // Ignora erros de extensões
    if (event.filename) {
        for (const pattern of IGNORE_PATTERNS) {
            if (pattern.test(event.filename)) {
                return true;
            }
        }
    }

    // Ignora erros sem mensagem relevante
    if (!event.message || event.message.trim().length < 3) {
        return true;
    }

    return false;
}

/**
 * Adiciona log ao buffer e envia quando necessário
 */
function addLog(type, message, data = null, severity = 'info') {
    // Verifica se está habilitado
    if (!isEnabled) return;
    
    // Verifica limite de logs por sessão
    if (totalLogsInSession >= maxLogsPerSession) {
        if (debugMode) {
            console.warn('[Console Tracker] Limite de logs por sessão atingido');
        }
        return;
    }

    if (debugMode) {
        console.log('[Console Tracker] Adicionando log:', { type, message, severity });
    }

    const log = {
        type,
        message: typeof message === 'string' ? message : JSON.stringify(message),
        data: data ? (typeof data === 'string' ? data : JSON.stringify(data)) : null,
        severity,
        context: collectContext(),
    };

    // Adiciona ao buffer
    logBuffer.push(log);

    if (debugMode) {
        console.log('[Console Tracker] Buffer size:', logBuffer.length);
    }

    // Debounce: aguarda um pouco antes de enviar (evita muitas requisições)
    if (debounceTimer) {
        clearTimeout(debounceTimer);
    }
    
    // Se o buffer estiver cheio, envia imediatamente
    if (logBuffer.length >= bufferSize) {
        flushLogs();
    } else {
        // Caso contrário, agenda envio com debounce
        debounceTimer = setTimeout(() => {
            if (logBuffer.length > 0) {
                flushLogs();
            }
        }, debounceDelay);
    }
}

/**
 * Envia todos os logs do buffer em batch
 */
function flushLogs() {
    if (logBuffer.length === 0 || !isEnabled) return;
    
    // Limpa o debounce timer
    if (debounceTimer) {
        clearTimeout(debounceTimer);
        debounceTimer = null;
    }

    const logsToSend = [...logBuffer];
    logBuffer = [];

    // Envia todos os logs em uma única requisição (batch)
    sendLogsBatch(logsToSend);
}

/**
 * Intercepta console.error
 */
function interceptConsoleError() {
    const originalError = console.error;
    console.error = function(...args) {
        // Chama o método original
        originalError.apply(console, args);

        // Captura o erro
        const message = args.map(arg => {
            if (arg instanceof Error) {
                return arg.message;
            }
            return String(arg);
        }).join(' ');

        const error = args.find(arg => arg instanceof Error);
        const stack = error ? getStackTrace(error) : null;

        // Ignora logs de extensões e serviços externos
        if (shouldIgnoreLog(message, stack)) {
            if (debugMode) {
                console.log('[Console Tracker] Log ignorado:', message);
            }
            return;
        }

        addLog('console.error', message, stack, 'error');
    };
}

/**
 * Intercepta console.warn
 */
function interceptConsoleWarn() {
    const originalWarn = console.warn;
    console.warn = function(...args) {
        // Chama o método original
        originalWarn.apply(console, args);

        // Captura o warning
        const message = args.map(arg => String(arg)).join(' ');

        // Ignora logs de extensões e serviços externos
        if (shouldIgnoreLog(message)) {
            return;
        }

        addLog('console.warn', message, null, 'warning');
    };
}

/**
 * Captura erros não tratados
 */
function captureUnhandledErrors() {
    window.addEventListener('error', (event) => {
        // Ignora erros de extensões
        if (shouldIgnoreUnhandledError(event)) {
            return;
        }

        const log = {
            type: 'unhandled_error',
            message: event.message || 'Erro não tratado',
            data: JSON.stringify({
                filename: event.filename,
                lineno: event.lineno,
                colno: event.colno,
                stack: event.error ? getStackTrace(event.error) : null,
            }),
            severity: 'error',
            context: collectContext(),
        };

        // Usa addLog para aproveitar buffer e rate limiting
        addLog(log.type, log.message, log.data, log.severity);
    });
}

/**
 * Captura promessas rejeitadas não tratadas
 */
function captureUnhandledRejections() {
    window.addEventListener('unhandledrejection', (event) => {
        const error = event.reason;
        const message = error instanceof Error ? error.message : String(error);
        const stack = error instanceof Error ? getStackTrace(error) : JSON.stringify(error);

        // Ignora rejeições de extensões
        if (shouldIgnoreLog(message, stack)) {
            return;
        }

        // Usa addLog para aproveitar buffer e rate limiting
        addLog('unhandled_rejection', `Promise rejeitada: ${message}`, stack, 'error');
    });
}

/**
 * Inicializa o rastreamento de console
 */
export function initConsoleTracker(options = {}) {
    if (isInitialized) {
        console.warn('Console tracker já foi inicializado');
        return;
    }

    // Verifica se está desabilitado (útil para produção)
    if (options.enabled === false) {
        isEnabled = false;
        if (options.debug) {
            console.log('[Console Tracker] Desabilitado por opção');
        }
        return;
    }

    // Configura opções
    bufferSize = options.bufferSize || bufferSize;
    flushInterval = options.flushInterval || flushInterval;
    debugMode = options.debug || false;
    maxLogsPerMinute = options.maxLogsPerMinute || maxLogsPerMinute;
    maxLogsPerSession = options.maxLogsPerSession || maxLogsPerSession;
    debounceDelay = options.debounceDelay || debounceDelay;
    
    // Configura origens permitidas (padrão: domínio atual)
    if (options.allowedOrigins) {
        allowedOrigins = options.allowedOrigins;
    } else {
        allowedOrigins = [window.location.origin];
    }

    // Intercepta métodos do console
    interceptConsoleError();
    interceptConsoleWarn();

    // Captura erros globais
    captureUnhandledErrors();
    captureUnhandledRejections();

    // Configura flush periódico
    flushTimer = setInterval(flushLogs, flushInterval);

    // Flush ao sair da página
    window.addEventListener('beforeunload', flushLogs);
    window.addEventListener('pagehide', flushLogs);

    isInitialized = true;
    
    if (debugMode) {
        console.log('[Console Tracker] Inicializado com otimizações de performance');
        console.log('[Console Tracker] Config:', {
            bufferSize,
            flushInterval: `${flushInterval / 1000}s`,
            maxLogsPerMinute,
            maxLogsPerSession,
            debounceDelay: `${debounceDelay / 1000}s`,
        });
    } else {
        console.log('Console tracker inicializado');
    }
}

/**
 * Habilita/desabilita modo debug
 */
export function setDebugMode(enabled) {
    debugMode = enabled;
    if (enabled) {
        console.log('[Console Tracker] Modo debug ativado');
    }
}

/**
 * Para o rastreamento (útil para testes)
 */
export function stopConsoleTracker() {
    if (!isInitialized) return;

    if (flushTimer) {
        clearInterval(flushTimer);
        flushTimer = null;
    }

    if (debounceTimer) {
        clearTimeout(debounceTimer);
        debounceTimer = null;
    }

    flushLogs();
    isInitialized = false;
    isEnabled = false;
}

/**
 * Habilita/desabilita o tracker dinamicamente
 */
export function setTrackerEnabled(enabled) {
    isEnabled = enabled;
    if (debugMode) {
        console.log('[Console Tracker]', enabled ? 'Habilitado' : 'Desabilitado');
    }
}

// O rastreamento deve ser inicializado explicitamente via initConsoleTracker()
// Veja app.js para a inicialização
