/**
 * Helper para carregar fixtures JSON durante desenvolvimento
 * Útil quando a API não está disponível
 */

const FIXTURES_PATH = '/js/fixtures/';

/**
 * Carrega uma fixture JSON
 * 
 * @param {string} fixtureName - Nome da fixture (sem extensão .json)
 * @returns {Promise<object>} - Dados da fixture
 */
export async function loadFixture(fixtureName) {
    try {
        const response = await fetch(`${FIXTURES_PATH}${fixtureName}.json`);
        if (!response.ok) {
            throw new Error(`Fixture ${fixtureName} não encontrada`);
        }
        return await response.json();
    } catch (error) {
        console.warn(`Erro ao carregar fixture ${fixtureName}:`, error);
        throw error;
    }
}

/**
 * Verifica se deve usar fixtures (modo de desenvolvimento)
 * Define via localStorage ou variável de ambiente
 */
export function shouldUseFixtures() {
    // Pode ser controlado via localStorage ou variável global
    if (typeof window !== 'undefined') {
        return window.USE_FIXTURES === true || localStorage.getItem('useFixtures') === 'true';
    }
    return false;
}

/**
 * Wrapper para apiFetch que usa fixtures quando necessário
 * 
 * @param {string} url - URL da API
 * @param {object} options - Opções do fetch
 * @param {string} fixtureName - Nome da fixture a usar (opcional)
 */
export async function apiFetchWithFixtures(url, options = {}, fixtureName = null) {
    // Se não estiver em modo fixtures, usa apiFetch normal
    if (!shouldUseFixtures() || !fixtureName) {
        const { apiFetch } = await import('./api');
        return apiFetch(url, options);
    }

    // Simula delay de rede
    await new Promise(resolve => setTimeout(resolve, 300));

    // Carrega fixture
    const fixture = await loadFixture(fixtureName);
    
    // Simula resposta da API
    return { data: fixture.data || fixture };
}

// Disponibiliza globalmente
if (typeof window !== 'undefined') {
    window.loadFixture = loadFixture;
    window.shouldUseFixtures = shouldUseFixtures;
}
