/**
 * Helpers para formulários
 * Mapeia erros da API (422) para campos do formulário
 */

/**
 * Mapeia erros da API para campos do formulário
 * 
 * @param {object} errors - Objeto de erros da API { field: ['message1', 'message2'] }
 * @param {HTMLElement} formElement - Elemento do formulário
 */
export function mapApiErrorsToFields(errors, formElement) {
    if (!errors || !formElement) return;

    // Limpa erros anteriores
    clearFormErrors(formElement);

    // Mapeia cada erro para o campo correspondente
    Object.keys(errors).forEach(fieldName => {
        const field = formElement.querySelector(`[name="${fieldName}"], [name="${fieldName}[]"]`);
        if (!field) return;

        const errorMessages = Array.isArray(errors[fieldName]) 
            ? errors[fieldName] 
            : [errors[fieldName]];

        const firstError = errorMessages[0];

        // Adiciona classe de erro ao campo
        field.classList.add('border-red-500');
        field.classList.remove('border-gray-300');

        // Cria ou atualiza mensagem de erro
        let errorElement = field.parentElement.querySelector('.text-red-600');
        if (!errorElement) {
            errorElement = document.createElement('p');
            errorElement.className = 'mt-1 text-sm text-red-600';
            field.parentElement.appendChild(errorElement);
        }
        errorElement.textContent = firstError;
    });
}

/**
 * Limpa erros do formulário
 * 
 * @param {HTMLElement} formElement - Elemento do formulário
 */
export function clearFormErrors(formElement) {
    if (!formElement) return;

    // Remove classes de erro dos campos
    const errorFields = formElement.querySelectorAll('.border-red-500');
    errorFields.forEach(field => {
        field.classList.remove('border-red-500');
        field.classList.add('border-gray-300');
    });

    // Remove mensagens de erro
    const errorMessages = formElement.querySelectorAll('.text-red-600');
    errorMessages.forEach(msg => msg.remove());
}

/**
 * Valida formulário antes de submeter
 * 
 * @param {HTMLElement} formElement - Elemento do formulário
 * @returns {boolean} - true se válido
 */
export function validateForm(formElement) {
    if (!formElement) return false;

    clearFormErrors(formElement);
    let isValid = true;

    // Valida campos obrigatórios
    const requiredFields = formElement.querySelectorAll('[required]');
    requiredFields.forEach(field => {
        if (!field.value || (field.type === 'checkbox' && !field.checked)) {
            field.classList.add('border-red-500');
            field.classList.remove('border-gray-300');
            
            let errorElement = field.parentElement.querySelector('.text-red-600');
            if (!errorElement) {
                errorElement = document.createElement('p');
                errorElement.className = 'mt-1 text-sm text-red-600';
                field.parentElement.appendChild(errorElement);
            }
            errorElement.textContent = 'Este campo é obrigatório';
            
            isValid = false;
        }
    });

    return isValid;
}

// Disponibiliza globalmente
if (typeof window !== 'undefined') {
    window.mapApiErrorsToFields = mapApiErrorsToFields;
    window.clearFormErrors = clearFormErrors;
    window.validateForm = validateForm;
}
