@props([
    'name',
    'label' => null,
    'options' => [], // Array de objetos com id e name
    'value' => [], // Array de IDs selecionados
    'placeholder' => 'Selecione opções',
    'required' => false,
    'error' => null,
    'searchUrl' => null, // URL para busca remota (se > 200 itens)
    'searchParam' => 'q', // Parâmetro de busca
])

@php
    $hasError = $error || ($errors->has($name) ?? false);
    $errorMessage = $error ?? ($errors->first($name) ?? null);
    $inputId = $attributes->get('id', $name);
    $selectedIds = old($name . '[]', $value ?? []);
    $useRemoteSearch = $searchUrl && count($options) > 200;
    $multiselectId = $inputId . '-multiselect';
@endphp

<div class="mb-4" 
     x-data="{
         open: false,
         selectedIds: {{ json_encode($selectedIds) }},
         allOptions: {{ json_encode($options) }},
         filteredOptions: {{ $useRemoteSearch ? '[]' : json_encode($options) }},
         selectedItems: [],
         searchQuery: '',
         searching: false,
         useRemoteSearch: {{ $useRemoteSearch ? 'true' : 'false' }},
         searchUrl: '{{ $searchUrl }}',
         searchParam: '{{ $searchParam }}',
         
         init() {
             this.updateSelectedItems();
         },
         
         updateSelectedItems() {
             this.selectedItems = this.allOptions.filter(opt => 
                 this.selectedIds.includes(opt.id)
             );
         },
         
         isSelected(id) {
             return this.selectedIds.includes(id);
         },
         
         toggleItem(option) {
             const index = this.selectedIds.indexOf(option.id);
             if (index > -1) {
                 this.selectedIds.splice(index, 1);
             } else {
                 this.selectedIds.push(option.id);
             }
             this.updateSelectedItems();
         },
         
         removeItem(id) {
             const index = this.selectedIds.indexOf(id);
             if (index > -1) {
                 this.selectedIds.splice(index, 1);
                 this.updateSelectedItems();
             }
         },
         
         async searchRemote() {
             if (!this.searchUrl || !this.searchQuery) {
                 this.filteredOptions = [];
                 return;
             }
             
             this.searching = true;
             try {
                 const url = `${this.searchUrl}?${this.searchParam}=${encodeURIComponent(this.searchQuery)}`;
                 const response = await window.apiFetch(url);
                 this.filteredOptions = response.data.data || response.data || [];
             } catch (error) {
                 console.error('Erro na busca:', error);
                 this.filteredOptions = [];
             } finally {
                 this.searching = false;
             }
         }
     }"
    @if($label)
        <label class="block mb-2 text-sm font-medium text-gray-900">
            {{ $label }}
            @if($required)
                <span class="text-red-500">*</span>
            @endif
        </label>
    @endif
    
    <!-- Hidden input para valores selecionados -->
    <template x-for="id in selectedIds" :key="id">
        <input type="hidden" name="{{ $name }}[]" :value="id">
    </template>
    
    <!-- Multiselect Container -->
    <div class="relative">
        <!-- Selected Items Display -->
        <div 
            @click="open = !open"
            class="bg-gray-50 border {{ $hasError ? 'border-red-500' : 'border-gray-300' }} text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full p-2.5 min-h-[42px] cursor-pointer flex flex-wrap gap-2"
        >
            <template x-if="selectedItems.length === 0">
                <span class="text-gray-500">{{ $placeholder }}</span>
            </template>
            <template x-for="item in selectedItems" :key="item.id">
                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                    <span x-text="item.name"></span>
                    <button 
                        type="button"
                        @click.stop="removeItem(item.id)"
                        class="ml-1 inline-flex items-center justify-center w-4 h-4 rounded-full hover:bg-blue-200"
                    >
                        <svg class="w-3 h-3" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clip-rule="evenodd"></path>
                        </svg>
                    </button>
                </span>
            </template>
        </div>
        
        <!-- Dropdown -->
        <div 
            x-show="open"
            @click.away="open = false"
            x-transition:enter="transition ease-out duration-100"
            x-transition:enter-start="transform opacity-0 scale-95"
            x-transition:enter-end="transform opacity-100 scale-100"
            x-transition:leave="transition ease-in duration-75"
            x-transition:leave-start="transform opacity-100 scale-100"
            x-transition:leave-end="transform opacity-0 scale-95"
            class="absolute z-50 w-full mt-1 bg-white border border-gray-300 rounded-lg shadow-lg max-h-60 overflow-auto"
            style="display: none;"
        >
            <!-- Search Input (se busca remota) -->
            <template x-if="useRemoteSearch">
                <div class="p-2 border-b border-gray-200">
                    <input 
                        type="text"
                        x-model="searchQuery"
                        @input.debounce.300ms="searchRemote()"
                        placeholder="Buscar..."
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500"
                    >
                    <div x-show="searching" class="mt-2 text-sm text-gray-500">Buscando...</div>
                </div>
            </template>
            
            <!-- Options List -->
            <template x-if="!useRemoteSearch || filteredOptions.length > 0">
                <div>
                    <template x-for="option in (useRemoteSearch ? filteredOptions : allOptions)" :key="option.id">
                        <div 
                            @click="toggleItem(option)"
                            class="px-4 py-2 hover:bg-gray-100 cursor-pointer flex items-center"
                            :class="{ 'bg-blue-50': isSelected(option.id) }"
                        >
                            <input 
                                type="checkbox" 
                                :checked="isSelected(option.id)"
                                class="mr-2"
                                readonly
                            >
                            <span x-text="option.name"></span>
                        </div>
                    </template>
                </div>
            </template>
            
            <template x-if="useRemoteSearch && filteredOptions.length === 0 && !searching">
                <div class="px-4 py-2 text-sm text-gray-500">Nenhum resultado encontrado</div>
            </template>
        </div>
    </div>
    
    @if($hasError)
        <p class="mt-1 text-sm text-red-600">{{ $errorMessage }}</p>
    @endif
</div>

