<x-app-layout>
    <x-slot name="page-title">Logs do Console</x-slot>

    <x-slot name="breadcrumbs">
        <li><a href="{{ route('dashboard') }}" class="hover:text-gray-900">Dashboard</a></li>
        <li><span class="text-gray-500">/</span></li>
        <li><span class="text-gray-700">Logs do Console</span></li>
    </x-slot>

    <div x-data="consoleLogs()" class="space-y-6">
        <!-- Estatísticas -->
        <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm font-medium text-gray-600">Total de Logs</p>
                        <p class="text-2xl font-semibold text-gray-900" x-text="stats.total || 0"></p>
                    </div>
                    <div class="p-3 bg-blue-100 rounded-lg">
                        <svg class="w-6 h-6 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
                            <path d="M2 6a2 2 0 012-2h6a2 2 0 012 2v8a2 2 0 01-2 2H4a2 2 0 01-2-2V6zM14.553 7.106A1 1 0 0014 8v4a1 1 0 00.553.894l2 1A1 1 0 0018 13V7a1 1 0 00-1.447-.894l-2 1z"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm font-medium text-gray-600">Erros (24h)</p>
                        <p class="text-2xl font-semibold text-red-600" x-text="stats.recent_errors || 0"></p>
                    </div>
                    <div class="p-3 bg-red-100 rounded-lg">
                        <svg class="w-6 h-6 text-red-600" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm font-medium text-gray-600">Avisos (24h)</p>
                        <p class="text-2xl font-semibold text-yellow-600" x-text="stats.recent_warnings || 0"></p>
                    </div>
                    <div class="p-3 bg-yellow-100 rounded-lg">
                        <svg class="w-6 h-6 text-yellow-600" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm font-medium text-gray-600">Resolução</p>
                        <div class="mt-2 space-y-1 text-xs">
                            <div class="flex justify-between">
                                <span class="text-gray-600">Resolvidos:</span>
                                <span class="font-semibold text-green-600" x-text="stats.resolved || 0"></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-600">Não Resolvidos:</span>
                                <span class="font-semibold text-red-600" x-text="stats.unresolved || 0"></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-600">Erros Não Resolvidos:</span>
                                <span class="font-semibold text-red-600" x-text="stats.unresolved_errors || 0"></span>
                            </div>
                        </div>
                    </div>
                    <div class="p-3 bg-blue-100 rounded-lg">
                        <svg class="w-6 h-6 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm font-medium text-gray-600">Por Severidade</p>
                        <div class="mt-2 space-y-1 text-xs">
                            <div class="flex justify-between">
                                <span class="text-gray-600">Erros:</span>
                                <span class="font-semibold text-red-600" x-text="(stats.by_severity && stats.by_severity.error) || 0"></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-600">Avisos:</span>
                                <span class="font-semibold text-yellow-600" x-text="(stats.by_severity && stats.by_severity.warning) || 0"></span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filtros -->
        <div class="bg-white rounded-lg shadow p-6">
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Tipo</label>
                    <select x-model="filters.type" @change="loadLogs()" class="w-full border-gray-300 rounded-md shadow-sm focus:border-blue-500 focus:ring-blue-500">
                        <option value="">Todos</option>
                        <option value="console.error">console.error</option>
                        <option value="console.warn">console.warn</option>
                        <option value="unhandled_error">Erro Não Tratado</option>
                        <option value="unhandled_rejection">Promise Rejeitada</option>
                    </select>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Severidade</label>
                    <select x-model="filters.severity" @change="loadLogs()" class="w-full border-gray-300 rounded-md shadow-sm focus:border-blue-500 focus:ring-blue-500">
                        <option value="">Todas</option>
                        <option value="error">Erro</option>
                        <option value="warning">Aviso</option>
                        <option value="info">Info</option>
                    </select>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Período</label>
                    <select x-model="filters.recent" @change="loadLogs()" class="w-full border-gray-300 rounded-md shadow-sm focus:border-blue-500 focus:ring-blue-500">
                        <option value="">Todos</option>
                        <option value="1">Última hora</option>
                        <option value="24">Últimas 24 horas</option>
                        <option value="168">Última semana</option>
                        <option value="720">Último mês</option>
                    </select>
                </div>
            </div>
            
            <!-- Filtros adicionais -->
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mt-4 pt-4 border-t border-gray-200">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                    <select x-model="filters.status" @change="loadLogs()" class="w-full border-gray-300 rounded-md shadow-sm focus:border-blue-500 focus:ring-blue-500">
                        <option value="">Todos</option>
                        <option value="unique">Apenas Únicos</option>
                        <option value="repeated">Apenas Repetidos</option>
                        <option value="obsolete">Apenas Obsoletos</option>
                        <option value="unresolved">Não Resolvidos</option>
                        <option value="resolved">Resolvidos</option>
                    </select>
                </div>

                <div class="flex items-end gap-2">
                    <button @click="copyLogs()" class="flex-1 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors" :class="{'bg-green-600 hover:bg-green-700': copySuccess}">
                        <svg class="w-5 h-5 inline mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z"></path>
                        </svg>
                        <span x-text="copySuccess ? 'Copiado!' : 'Copiar Logs'"></span>
                    </button>
                    <button @click="exportLogs('json')" class="flex-1 px-4 py-2 bg-gray-600 text-white rounded-md hover:bg-gray-700 transition-colors">
                        <svg class="w-5 h-5 inline mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                        </svg>
                        JSON
                    </button>
                    <button @click="exportLogs('csv')" class="flex-1 px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition-colors">
                        <svg class="w-5 h-5 inline mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                        </svg>
                        CSV
                    </button>
                </div>
            </div>
        </div>

        <!-- Tabela de Logs -->
        <div class="bg-white rounded-lg shadow overflow-hidden">
            <div class="px-6 py-4 border-b border-gray-200">
                <h2 class="text-lg font-semibold text-gray-900">Logs Capturados</h2>
            </div>

            <div x-show="loading" class="p-8 text-center">
                <div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                <p class="mt-2 text-gray-600">Carregando logs...</p>
            </div>

            <div x-show="!loading && logs.length === 0" class="p-8 text-center text-gray-500">
                Nenhum log encontrado com os filtros selecionados.
            </div>

            <div x-show="!loading && logs.length > 0" class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Data/Hora</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tipo</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Severidade</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Mensagem</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Usuário</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ações</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <template x-for="log in logs" :key="log.id">
                            <tr class="hover:bg-gray-50" :class="{'opacity-60': log.is_obsolete}">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex flex-col gap-1">
                                        <!-- Badge de repetição -->
                                        <template x-if="log.is_repeated && log.occurrence_count > 1">
                                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-orange-100 text-orange-800" 
                                                  :title="'Repetido ' + log.occurrence_count + ' vezes'">
                                                🔁 <span x-text="log.occurrence_count"></span>x
                                            </span>
                                        </template>
                                        <!-- Badge de único -->
                                        <template x-if="!log.is_repeated">
                                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800" title="Log único">
                                                ✓ Único
                                            </span>
                                        </template>
                                        <!-- Badge de obsoleto -->
                                        <template x-if="log.is_obsolete">
                                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-gray-100 text-gray-600" title="Log obsoleto (mais de 7 dias)">
                                                ⏰ Obsoleto
                                            </span>
                                        </template>
                                        <!-- Badge de resolvido -->
                                        <span x-show="log && log.is_resolved === true" 
                                              class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800" 
                                              :title="log && log.resolved_at ? 'Resolvido em ' + formatDate(log.resolved_at) + (log.resolver ? ' por ' + log.resolver.name : '') : 'Resolvido'">
                                            ✓ Resolvido
                                        </span>
                                        <!-- Badge de não resolvido -->
                                        <span x-show="log && (log.is_resolved === false || !log.is_resolved)" 
                                              class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-red-100 text-red-800" 
                                              title="Não resolvido">
                                            ⚠ Não Resolvido
                                        </span>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-900" x-text="formatDate(log.created_at)"></div>
                                    <template x-if="log.is_repeated && log.first_occurrence">
                                        <div class="text-xs text-gray-500 mt-1" 
                                             :title="'Primeira ocorrência: ' + formatDate(log.first_occurrence)">
                                            Primeira: <span x-text="formatDate(log.first_occurrence)"></span>
                                        </div>
                                    </template>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 text-blue-800" x-text="log.type"></span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span 
                                        class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full"
                                        :class="{
                                            'bg-red-100 text-red-800': log.severity === 'error',
                                            'bg-yellow-100 text-yellow-800': log.severity === 'warning',
                                            'bg-blue-100 text-blue-800': log.severity === 'info'
                                        }"
                                        x-text="log.severity"
                                    ></span>
                                </td>
                                <td class="px-6 py-4 text-sm text-gray-900">
                                    <div class="max-w-md truncate" x-text="log.message" :title="log.message"></div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500" x-text="log.user ? log.user.name : '-'"></td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <div class="flex items-center gap-2">
                                        <button @click="showLogDetails(log)" class="text-blue-600 hover:text-blue-900">Detalhes</button>
                                        <button 
                                            @click="toggleResolved(log)" 
                                            :class="log && log.is_resolved ? 'text-yellow-600 hover:text-yellow-900' : 'text-green-600 hover:text-green-900'"
                                            :title="log && log.is_resolved ? 'Marcar como não resolvido' : 'Marcar como resolvido'"
                                        >
                                            <span x-show="log && log.is_resolved">Desmarcar</span>
                                            <span x-show="log && !log.is_resolved">Resolver</span>
                                        </button>
                                        <button @click="deleteLog(log.id)" class="text-red-600 hover:text-red-900">Excluir</button>
                                    </div>
                                </td>
                            </tr>
                        </template>
                    </tbody>
                </table>
            </div>

            <!-- Paginação -->
            <div x-show="!loading && logs.length > 0" class="px-6 py-4 border-t border-gray-200 flex items-center justify-between">
                <div class="text-sm text-gray-700">
                    Mostrando <span x-text="pagination.from || 0"></span> a <span x-text="pagination.to || 0"></span> de <span x-text="pagination.total || 0"></span> resultados
                </div>
                <div class="flex gap-2">
                    <button 
                        @click="loadLogs(pagination.prev_page_url)"
                        :disabled="!pagination.prev_page_url"
                        :class="{'opacity-50 cursor-not-allowed': !pagination.prev_page_url}"
                        class="px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                    >
                        Anterior
                    </button>
                    <button 
                        @click="loadLogs(pagination.next_page_url)"
                        :disabled="!pagination.next_page_url"
                        :class="{'opacity-50 cursor-not-allowed': !pagination.next_page_url}"
                        class="px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                    >
                        Próxima
                    </button>
                </div>
            </div>
        </div>

        <!-- Modal de Detalhes -->
        <div 
            x-show="showModal"
            @click.away="showModal = false"
            x-transition:enter="transition ease-out duration-200"
            x-transition:enter-start="opacity-0"
            x-transition:enter-end="opacity-100"
            x-transition:leave="transition ease-in duration-150"
            x-transition:leave-start="opacity-100"
            x-transition:leave-end="opacity-0"
            class="fixed inset-0 bg-gray-900 bg-opacity-50 z-50 flex items-center justify-center p-4"
            style="display: none;"
        >
            <div class="bg-white rounded-lg shadow-xl max-w-4xl w-full max-h-[90vh] overflow-y-auto">
                <div class="px-6 py-4 border-b border-gray-200 flex items-center justify-between">
                    <h3 class="text-lg font-semibold text-gray-900">Detalhes do Log</h3>
                    <button @click="showModal = false" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div class="p-6 space-y-4" x-show="selectedLog">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Mensagem</label>
                        <div class="bg-gray-50 p-3 rounded-md text-sm font-mono" x-text="selectedLog?.message"></div>
                    </div>
                    <div x-show="selectedLog?.data">
                        <label class="block text-sm font-medium text-gray-700 mb-1">Stack Trace / Dados</label>
                        <pre class="bg-gray-50 p-3 rounded-md text-xs font-mono overflow-x-auto" x-text="selectedLog?.data"></pre>
                    </div>
                    <div x-show="selectedLog?.context">
                        <label class="block text-sm font-medium text-gray-700 mb-1">Contexto</label>
                        <pre class="bg-gray-50 p-3 rounded-md text-xs font-mono overflow-x-auto" x-text="JSON.stringify(selectedLog?.context, null, 2)"></pre>
                    </div>
                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Tipo</label>
                            <div class="text-sm text-gray-900" x-text="selectedLog?.type"></div>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Severidade</label>
                            <div class="text-sm text-gray-900" x-text="selectedLog?.severity"></div>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Data/Hora</label>
                            <div class="text-sm text-gray-900" x-text="formatDate(selectedLog?.created_at)"></div>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Usuário</label>
                            <div class="text-sm text-gray-900" x-text="selectedLog?.user ? selectedLog.user.name : 'Não autenticado'"></div>
                        </div>
                        <template x-if="selectedLog?.is_repeated">
                            <div class="col-span-2">
                                <label class="block text-sm font-medium text-gray-700 mb-1">Informações de Repetição</label>
                                <div class="bg-orange-50 border border-orange-200 rounded-md p-3 space-y-1">
                                    <div class="text-sm">
                                        <span class="font-medium text-orange-800">Ocorrências:</span>
                                        <span class="text-orange-700" x-text="selectedLog?.occurrence_count || 1"></span>
                                    </div>
                                    <template x-if="selectedLog?.first_occurrence">
                                        <div class="text-sm">
                                            <span class="font-medium text-orange-800">Primeira ocorrência:</span>
                                            <span class="text-orange-700" x-text="formatDate(selectedLog.first_occurrence)"></span>
                                        </div>
                                    </template>
                                    <div class="text-sm">
                                        <span class="font-medium text-orange-800">Última ocorrência:</span>
                                        <span class="text-orange-700" x-text="formatDate(selectedLog?.created_at)"></span>
                                    </div>
                                </div>
                            </div>
                        </template>
                        <template x-if="selectedLog?.is_obsolete">
                            <div class="col-span-2">
                                <div class="bg-gray-50 border border-gray-200 rounded-md p-3">
                                    <div class="text-sm text-gray-600">
                                        <span class="font-medium">⚠️ Log Obsoleto:</span> Este log tem mais de 7 dias e pode não ser mais relevante.
                                    </div>
                                </div>
                            </div>
                        </template>
                        <template x-if="selectedLog?.is_resolved">
                            <div class="col-span-2">
                                <label class="block text-sm font-medium text-gray-700 mb-1">Informações de Resolução</label>
                                <div class="bg-green-50 border border-green-200 rounded-md p-3 space-y-1">
                                    <div class="text-sm">
                                        <span class="font-medium text-green-800">Status:</span>
                                        <span class="text-green-700"> Resolvido</span>
                                    </div>
                                    <template x-if="selectedLog?.resolved_at">
                                        <div class="text-sm">
                                            <span class="font-medium text-green-800">Resolvido em:</span>
                                            <span class="text-green-700" x-text="formatDate(selectedLog.resolved_at)"></span>
                                        </div>
                                    </template>
                                    <template x-if="selectedLog?.resolver">
                                        <div class="text-sm">
                                            <span class="font-medium text-green-800">Resolvido por:</span>
                                            <span class="text-green-700" x-text="selectedLog.resolver.name"></span>
                                        </div>
                                    </template>
                                    <template x-if="selectedLog?.resolution_notes">
                                        <div class="text-sm mt-2">
                                            <span class="font-medium text-green-800">Notas:</span>
                                            <div class="text-green-700 mt-1" x-text="selectedLog.resolution_notes"></div>
                                        </div>
                                    </template>
                                </div>
                            </div>
                        </template>
                    </div>
                </div>
            </div>
        </div>
    </div>

    @push('scripts')
    <script>
        function consoleLogs() {
            return {
                logs: [],
                stats: {},
                loading: false,
                showModal: false,
                selectedLog: null,
                pagination: {},
                copySuccess: false,
                filters: {
                    type: '',
                    severity: '',
                    recent: '',
                    status: ''
                },
                
                async init() {
                    await this.loadStats();
                    await this.loadLogs();
                },

                async loadStats() {
                    try {
                        const response = await fetch('/api/console-logs/stats', {
                            headers: {
                                'Accept': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            },
                            credentials: 'same-origin'
                        });
                        const data = await response.json();
                        this.stats = data.data || {};
                    } catch (error) {
                        console.error('Erro ao carregar estatísticas:', error);
                    }
                },

                async loadLogs(url = null) {
                    this.loading = true;
                    try {
                        const baseUrl = url || '/api/console-logs';
                        const params = new URLSearchParams();
                        
                        if (this.filters.type) params.append('type', this.filters.type);
                        if (this.filters.severity) params.append('severity', this.filters.severity);
                        if (this.filters.recent) params.append('recent', this.filters.recent);
                        if (this.filters.status) params.append('status', this.filters.status);
                        
                        const queryString = params.toString();
                        const finalUrl = url || (baseUrl + (queryString ? '?' + queryString : ''));
                        
                        const response = await fetch(finalUrl, {
                            headers: {
                                'Accept': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            },
                            credentials: 'same-origin'
                        });
                        
                        const data = await response.json();
                        this.logs = data.data || [];
                        this.pagination = {
                            current_page: data.current_page,
                            from: data.from,
                            to: data.to,
                            total: data.total,
                            per_page: data.per_page,
                            last_page: data.last_page,
                            prev_page_url: data.prev_page_url,
                            next_page_url: data.next_page_url
                        };
                    } catch (error) {
                        console.error('Erro ao carregar logs:', error);
                        showToast('Erro ao carregar logs', 'error');
                    } finally {
                        this.loading = false;
                    }
                },

                showLogDetails(log) {
                    this.selectedLog = log;
                    this.showModal = true;
                },

                async deleteLog(id) {
                    if (!confirm('Tem certeza que deseja excluir este log?')) {
                        return;
                    }

                    try {
                        const response = await fetch(`/api/console-logs/${id}`, {
                            method: 'DELETE',
                            headers: {
                                'Accept': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            },
                            credentials: 'same-origin'
                        });

                        if (response.ok) {
                            showToast('Log excluído com sucesso', 'success');
                            await this.loadLogs();
                            await this.loadStats();
                        } else {
                            showToast('Erro ao excluir log', 'error');
                        }
                    } catch (error) {
                        console.error('Erro ao excluir log:', error);
                        showToast('Erro ao excluir log', 'error');
                    }
                },

                async copyLogs() {
                    try {
                        // Busca todos os logs com os filtros atuais (sem paginação)
                        const params = new URLSearchParams();
                        if (this.filters.type) params.append('type', this.filters.type);
                        if (this.filters.severity) params.append('severity', this.filters.severity);
                        if (this.filters.recent) params.append('recent', this.filters.recent);
                        if (this.filters.status) params.append('status', this.filters.status);
                        params.append('per_page', '1000'); // Busca até 1000 logs
                        
                        const response = await fetch(`/api/console-logs?${params.toString()}`, {
                            headers: {
                                'Accept': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            },
                            credentials: 'same-origin'
                        });
                        
                        const data = await response.json();
                        const logsToCopy = data.data || [];
                        
                        // Formata os logs em JSON para facilitar colar
                        const logsJson = JSON.stringify({
                            total: logsToCopy.length,
                            filters: this.filters,
                            timestamp: new Date().toISOString(),
                            logs: logsToCopy
                        }, null, 2);
                        
                        // Copia para a área de transferência com fallback
                        try {
                            // Tenta usar a API moderna do clipboard
                            if (navigator.clipboard && window.isSecureContext) {
                                await navigator.clipboard.writeText(logsJson);
                            } else {
                                // Fallback: usa textarea temporário
                                throw new Error('Clipboard API não disponível');
                            }
                        } catch (clipboardError) {
                            // Fallback: cria textarea temporário e seleciona
                            const textarea = document.createElement('textarea');
                            textarea.value = logsJson;
                            textarea.style.position = 'fixed';
                            textarea.style.left = '-999999px';
                            textarea.style.top = '-999999px';
                            document.body.appendChild(textarea);
                            textarea.focus();
                            textarea.select();
                            
                            try {
                                const successful = document.execCommand('copy');
                                if (!successful) {
                                    throw new Error('Falha ao copiar');
                                }
                            } finally {
                                document.body.removeChild(textarea);
                            }
                        }
                        
                        // Feedback visual
                        this.copySuccess = true;
                        showToast(`Copiados ${logsToCopy.length} logs para a área de transferência`, 'success');
                        
                        // Reseta o feedback após 2 segundos
                        setTimeout(() => {
                            this.copySuccess = false;
                        }, 2000);
                    } catch (error) {
                        console.error('Erro ao copiar logs:', error);
                        showToast('Erro ao copiar logs: ' + error.message, 'error');
                    }
                },

                exportLogs(format = 'json') {
                    const params = new URLSearchParams();
                    if (this.filters.type) params.append('type', this.filters.type);
                    if (this.filters.severity) params.append('severity', this.filters.severity);
                    if (this.filters.recent) params.append('recent', this.filters.recent);
                    params.append('export', '1');
                    params.append('format', format);
                    
                    const queryString = params.toString();
                    const url = '/api/console-logs?' + queryString;
                    
                    // Cria um link temporário para download
                    const link = document.createElement('a');
                    link.href = url;
                    link.download = `console-logs-${new Date().toISOString().split('T')[0]}.${format}`;
                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);
                },

                formatDate(dateString) {
                    if (!dateString) return '-';
                    const date = new Date(dateString);
                    return date.toLocaleString('pt-BR', {
                        day: '2-digit',
                        month: '2-digit',
                        year: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit',
                        second: '2-digit'
                    });
                }
            }
        }
    </script>
    @endpush
</x-app-layout>
