<x-app-layout>
    <x-slot name="page-title">Episódios</x-slot>

    <x-slot name="breadcrumbs">
        <li><a href="{{ route('dashboard') }}" class="hover:text-gray-900">Dashboard</a></li>
        <li class="text-gray-500">/</li>
        <li class="text-gray-900">Episódios</li>
    </x-slot>

    <div class="space-y-6" x-data="episodesList()">
        <!-- Header com CTA -->
        <div class="flex items-center justify-between">
            <h1 class="text-2xl font-bold text-gray-900">Episódios</h1>
            <a href="{{ route('episodes.create') }}" class="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                </svg>
                Novo Episódio
            </a>
        </div>

        <!-- Filtros -->
        <x-card>
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <!-- Filtro Status -->
                <div>
                    <label class="block mb-2 text-sm font-medium text-gray-900">Status</label>
                    <select 
                        x-model="filters.status"
                        @change="loadEpisodes()"
                        class="bg-gray-50 border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full p-2.5"
                    >
                        <option value="">Todos</option>
                        @foreach($statuses as $status)
                            <option value="{{ $status }}">{{ ucfirst($status) }}</option>
                        @endforeach
                    </select>
                </div>

                <!-- Filtro Tema -->
                <div>
                    <label class="block mb-2 text-sm font-medium text-gray-900">Tema</label>
                    <select 
                        x-model="filters.theme_id"
                        @change="loadEpisodes()"
                        class="bg-gray-50 border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full p-2.5"
                    >
                        <option value="">Todos</option>
                        @foreach($themes as $theme)
                            <option value="{{ $theme->id }}">{{ $theme->name }}</option>
                        @endforeach
                    </select>
                </div>

                <!-- Filtro Convidado -->
                <div>
                    <label class="block mb-2 text-sm font-medium text-gray-900">Convidado</label>
                    <select 
                        x-model="filters.guest_id"
                        @change="loadEpisodes()"
                        class="bg-gray-50 border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full p-2.5"
                    >
                        <option value="">Todos</option>
                        @foreach($guests as $guest)
                            <option value="{{ $guest->id }}">{{ $guest->name }}</option>
                        @endforeach
                    </select>
                </div>

                <!-- Botão Limpar -->
                <div class="flex items-end">
                    <button 
                        @click="clearFilters()"
                        class="w-full px-4 py-2.5 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-100"
                    >
                        Limpar Filtros
                    </button>
                </div>
            </div>
        </x-card>

        <!-- Loading State -->
        <div x-show="loading" class="text-center py-12">
            <div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
            <p class="mt-2 text-gray-600">Carregando episódios...</p>
        </div>

        <!-- Empty State -->
        <x-empty-state 
            x-show="!loading && episodes.length === 0"
            title="Nenhum episódio encontrado"
            description="Comece criando seu primeiro episódio!"
            action-label="Criar Episódio"
            action-url="{{ route('episodes.create') }}"
            style="display: none;"
        />

        <!-- Tabela de Episódios -->
        <x-card :padding="false" x-show="!loading && episodes.length > 0" style="display: none;">
            <div class="overflow-x-auto">
                <table class="w-full text-sm text-left text-gray-500">
                    <thead class="text-xs text-gray-700 uppercase bg-gray-50">
                        <tr>
                            <th scope="col" class="px-6 py-3">Título</th>
                            <th scope="col" class="px-6 py-3">Status</th>
                            <th scope="col" class="px-6 py-3">Convidados</th>
                            <th scope="col" class="px-6 py-3">Temas</th>
                            <th scope="col" class="px-6 py-3">Gravação</th>
                            <th scope="col" class="px-6 py-3">Publicação</th>
                            <th scope="col" class="px-6 py-3">Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        <template x-for="episode in episodes" :key="episode.id">
                            <tr class="bg-white border-b hover:bg-gray-50">
                                <td class="px-6 py-4 font-medium text-gray-900" x-text="episode.title"></td>
                                <td class="px-6 py-4">
                                    <span x-html="getBadgeStatus(episode.status)"></span>
                                </td>
                                <td class="px-6 py-4">
                                    <template x-if="episode.guests && episode.guests.length > 0">
                                        <div class="flex flex-wrap gap-1">
                                            <template x-for="(guest, index) in episode.guests.slice(0, 2)" :key="guest.id">
                                                <span class="text-xs bg-gray-100 text-gray-800 px-2 py-1 rounded" x-text="guest.name"></span>
                                            </template>
                                            <template x-if="episode.guests.length > 2">
                                                <span class="text-xs text-gray-500" x-text="`+${episode.guests.length - 2}`"></span>
                                            </template>
                                        </div>
                                    </template>
                                    <template x-if="!episode.guests || episode.guests.length === 0">
                                        <span class="text-gray-400">—</span>
                                    </template>
                                </td>
                                <td class="px-6 py-4">
                                    <template x-if="episode.themes && episode.themes.length > 0">
                                        <div class="flex flex-wrap gap-1">
                                            <template x-for="theme in episode.themes.slice(0, 2)" :key="theme.id">
                                                <span class="text-xs bg-blue-100 text-blue-800 px-2 py-1 rounded" x-text="theme.name"></span>
                                            </template>
                                            <template x-if="episode.themes.length > 2">
                                                <span class="text-xs text-gray-500" x-text="`+${episode.themes.length - 2}`"></span>
                                            </template>
                                        </div>
                                    </template>
                                    <template x-if="!episode.themes || episode.themes.length === 0">
                                        <span class="text-gray-400">—</span>
                                    </template>
                                </td>
                                <td class="px-6 py-4" x-text="formatDate(episode.record_date)"></td>
                                <td class="px-6 py-4" x-text="formatDate(episode.publish_date)"></td>
                                <td class="px-6 py-4">
                                    <a 
                                        :href="`/episodes/${episode.id}`"
                                        class="text-blue-600 hover:text-blue-800 font-medium"
                                    >
                                        Abrir
                                    </a>
                                </td>
                            </tr>
                        </template>
                    </tbody>
                </table>
            </div>

            <!-- Paginação -->
            <div x-show="pagination && pagination.total > 0" class="px-6 py-4 border-t border-gray-200" style="display: none;">
                <div class="flex items-center justify-between">
                    <div class="text-sm text-gray-700">
                        Mostrando <span x-text="pagination?.from || 0"></span> a <span x-text="pagination?.to || 0"></span> de <span x-text="pagination?.total || 0"></span> episódios
                    </div>
                    <div class="flex gap-2">
                        <button 
                            @click="loadEpisodes((pagination?.current_page || 1) - 1)"
                            :disabled="!pagination || pagination.current_page === 1"
                            :class="(!pagination || pagination.current_page === 1) ? 'opacity-50 cursor-not-allowed' : ''"
                            class="px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-100"
                        >
                            Anterior
                        </button>
                        <button 
                            @click="loadEpisodes((pagination?.current_page || 1) + 1)"
                            :disabled="!pagination || pagination.current_page === pagination.last_page"
                            :class="(!pagination || pagination.current_page === pagination.last_page) ? 'opacity-50 cursor-not-allowed' : ''"
                            class="px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-100"
                        >
                            Próxima
                        </button>
                    </div>
                </div>
            </div>
        </x-card>
    </div>

    @push('scripts')
    <script>
        function episodesList() {
            return {
                loading: true,
                episodes: [],
                pagination: null,
                filters: {
                    status: '',
                    theme_id: '',
                    guest_id: '',
                },
                
                init() {
                    this.loadEpisodes();
                },
                
                async loadEpisodes(page = 1) {
                    this.loading = true;
                    
                    try {
                        const params = new URLSearchParams({
                            page: page,
                            ...this.filters
                        });
                        
                        // Remove parâmetros vazios
                        for (const [key, value] of params.entries()) {
                            if (!value) params.delete(key);
                        }
                        
                        const response = await window.apiFetch(`/api/episodes?${params.toString()}`);
                        
                        // A resposta vem como { data: [...], meta: {...} } quando há paginação
                        if (response.data && Array.isArray(response.data)) {
                            this.episodes = response.data;
                            this.pagination = response.meta || null;
                        } else {
                            this.episodes = [];
                            this.pagination = null;
                        }
                    } catch (error) {
                        console.error('Erro ao carregar episódios:', error);
                        window.showToast('Erro ao carregar episódios', 'error');
                    } finally {
                        this.loading = false;
                    }
                },
                
                clearFilters() {
                    this.filters = {
                        status: '',
                        theme_id: '',
                        guest_id: '',
                    };
                    this.loadEpisodes();
                },
                
                formatDate(date) {
                    if (!date) return '—';
                    return new Date(date).toLocaleDateString('pt-BR');
                },
                
                getBadgeStatus(status) {
                    const statusMap = {
                        'pauta': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">Pauta</span>',
                        'gravacao': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">Gravação</span>',
                        'edicao': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">Edição</span>',
                        'revisao': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">Revisão</span>',
                        'agendado': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">Agendado</span>',
                        'publicado': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">Publicado</span>',
                    };
                    return statusMap[status] || statusMap['pauta'];
                }
            }
        }
    </script>
    @endpush
</x-app-layout>
