@props(['episode'])

<div x-data="assetList({{ $episode->id }})">
    <!-- Formulário para adicionar asset -->
    <div class="mb-4 p-4 bg-gray-50 rounded-lg">
        <h3 class="text-sm font-medium text-gray-900 mb-3">Adicionar Asset</h3>
        <form @submit.prevent="addAsset" class="space-y-3">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-3">
                <div>
                    <label class="block mb-1 text-xs font-medium text-gray-700">Tipo</label>
                    <select 
                        x-model="newAsset.asset_type"
                        required
                        class="bg-white border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full p-2"
                    >
                        <option value="">Selecione...</option>
                        <option value="audio">Áudio</option>
                        <option value="video">Vídeo</option>
                        <option value="thumb">Thumb</option>
                        <option value="roteiro">Roteiro</option>
                        <option value="cortes">Cortes</option>
                        <option value="drive">Google Drive</option>
                        <option value="outros">Outros</option>
                    </select>
                </div>
                <div>
                    <label class="block mb-1 text-xs font-medium text-gray-700">Label (opcional)</label>
                    <input 
                        type="text"
                        x-model="newAsset.label"
                        placeholder="Ex: Versão final"
                        class="bg-white border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full p-2"
                    >
                </div>
                <div>
                    <label class="block mb-1 text-xs font-medium text-gray-700">URL <span class="text-red-500">*</span></label>
                    <input 
                        type="url"
                        x-model="newAsset.url"
                        placeholder="https://..."
                        required
                        class="bg-white border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full p-2"
                    >
                </div>
            </div>
            <div class="flex justify-end">
                <button 
                    type="submit"
                    :disabled="adding"
                    class="px-4 py-2 text-sm font-medium text-white bg-blue-600 rounded-lg hover:bg-blue-700 disabled:opacity-50"
                >
                    <span x-show="!adding">Adicionar</span>
                    <span x-show="adding" style="display: none;">Adicionando...</span>
                </button>
            </div>
        </form>
    </div>

    <!-- Lista de Assets -->
    <div class="space-y-2">
        <template x-if="loading">
            <div class="text-center py-4">
                <div class="inline-block animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div>
            </div>
        </template>

        <template x-if="!loading && assets.length === 0">
            <p class="text-gray-500 text-sm">Nenhum asset adicionado ainda.</p>
        </template>

        <template x-for="asset in assets" :key="asset.id">
            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors">
                <div class="flex-1">
                    <div class="flex items-center gap-2">
                        <span class="inline-flex items-center px-2 py-0.5 rounded text-xs font-medium bg-blue-100 text-blue-800" x-text="getAssetTypeLabel(asset.asset_type)"></span>
                        <template x-if="asset.label">
                            <span class="text-sm font-medium text-gray-900" x-text="asset.label"></span>
                        </template>
                    </div>
                    <a 
                        :href="asset.url" 
                        target="_blank"
                        class="text-sm text-blue-600 hover:text-blue-800 hover:underline mt-1 block truncate"
                        x-text="asset.url"
                    ></a>
                </div>
                <button 
                    @click="confirmDelete(asset)"
                    class="ml-4 text-red-600 hover:text-red-800 p-1"
                    title="Excluir"
                >
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                    </svg>
                </button>
            </div>
        </template>
    </div>
</div>

@push('scripts')
<script>
    function assetList(episodeId) {
        return {
            episodeId: episodeId,
            assets: [],
            loading: true,
            adding: false,
            newAsset: {
                asset_type: '',
                label: '',
                url: '',
            },
            
            init() {
                this.loadAssets();
            },
            
            async loadAssets() {
                this.loading = true;
                try {
                    const response = await window.apiFetch(`/api/episodes/${this.episodeId}/assets`);
                    this.assets = response.data || [];
                } catch (error) {
                    console.error('Erro ao carregar assets:', error);
                    window.showToast('Erro ao carregar assets', 'error');
                } finally {
                    this.loading = false;
                }
            },
            
            async addAsset() {
                this.adding = true;
                try {
                    const response = await window.apiFetch(`/api/episodes/${this.episodeId}/assets`, {
                        method: 'POST',
                        body: this.newAsset,
                    });
                    
                    // Adiciona à lista
                    this.assets.push(response.data);
                    
                    // Limpa formulário
                    this.newAsset = {
                        asset_type: '',
                        label: '',
                        url: '',
                    };
                    
                    window.showToast('Asset adicionado com sucesso!', 'success');
                } catch (error) {
                    console.error('Erro ao adicionar asset:', error);
                    if (error.errors) {
                        window.mapApiErrorsToFields(error.errors, document.querySelector('[x-data*="assetList"]'));
                        window.showToast('Corrija os erros do formulário', 'error');
                    } else {
                        window.showToast('Erro ao adicionar asset', 'error');
                    }
                } finally {
                    this.adding = false;
                }
            },
            
            confirmDelete(asset) {
                if (confirm(`Tem certeza que deseja excluir este asset?\n\n${asset.label || asset.asset_type}: ${asset.url}`)) {
                    this.deleteAsset(asset.id);
                }
            },
            
            async deleteAsset(assetId) {
                try {
                    await window.apiFetch(`/api/episodes/${this.episodeId}/assets/${assetId}`, {
                        method: 'DELETE',
                    });
                    
                    // Remove da lista
                    this.assets = this.assets.filter(a => a.id !== assetId);
                    
                    window.showToast('Asset removido com sucesso!', 'success');
                } catch (error) {
                    console.error('Erro ao remover asset:', error);
                    window.showToast('Erro ao remover asset', 'error');
                }
            },
            
            getAssetTypeLabel(type) {
                const labels = {
                    'audio': 'Áudio',
                    'video': 'Vídeo',
                    'thumb': 'Thumb',
                    'roteiro': 'Roteiro',
                    'cortes': 'Cortes',
                    'drive': 'Google Drive',
                    'outros': 'Outros',
                };
                return labels[type] || type;
            }
        }
    }
</script>
@endpush
