@props(['episode'])

<div x-data="statusTransition({{ $episode->id }}, '{{ $episode->status }}')">
    <!-- Botão Avançar Status -->
    <button 
        @click="loadTransitions()"
        :disabled="loadingTransitions"
        class="inline-flex items-center px-4 py-2 text-sm font-medium text-white bg-blue-600 rounded-lg hover:bg-blue-700 disabled:opacity-50"
    >
        Avançar Status
        <svg class="w-4 h-4 ml-2" fill="currentColor" viewBox="0 0 20 20">
            <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd"></path>
        </svg>
    </button>

    <!-- Dropdown de Transições -->
    <div 
        x-show="showDropdown"
        @click.away="showDropdown = false"
        x-transition:enter="transition ease-out duration-100"
        x-transition:enter-start="transform opacity-0 scale-95"
        x-transition:enter-end="transform opacity-100 scale-100"
        x-transition:leave="transition ease-in duration-75"
        x-transition:leave-start="transform opacity-100 scale-100"
        x-transition:leave-end="transform opacity-0 scale-95"
        class="absolute z-50 mt-2 right-0 w-56 bg-white rounded-lg shadow-lg border border-gray-200"
        style="display: none;"
    >
        <div class="p-2">
            <template x-if="loadingTransitions">
                <div class="text-center py-2 text-sm text-gray-500">Carregando...</div>
            </template>
            
            <template x-if="!loadingTransitions && transitions.length === 0">
                <div class="text-center py-2 text-sm text-gray-500">Nenhuma transição disponível</div>
            </template>
            
            <template x-for="transition in transitions" :key="transition">
                <button 
                    @click="confirmTransition(transition)"
                    class="w-full text-left px-3 py-2 text-sm text-gray-700 hover:bg-gray-100 rounded"
                >
                    <span x-text="getStatusLabel(transition)"></span>
                </button>
            </template>
        </div>
    </div>

    <!-- Modal de Bloqueios (Inline Blockers) -->
    <div 
        x-show="showBlockers"
        x-transition:enter="transition ease-out duration-200"
        x-transition:enter-start="opacity-0"
        x-transition:enter-end="opacity-100"
        x-transition:leave="transition ease-in duration-150"
        x-transition:leave-start="opacity-100"
        x-transition:leave-end="opacity-0"
        class="fixed inset-0 bg-gray-900 bg-opacity-50 z-50 flex items-center justify-center p-4"
        style="display: none;"
        @click.away="showBlockers = false"
    >
        <div 
            @click.stop
            class="bg-white rounded-lg shadow-xl max-w-md w-full p-6"
        >
            <h3 class="text-lg font-semibold text-gray-900 mb-4">Não é possível avançar o status</h3>
            
            <div class="space-y-3 mb-6">
                <template x-for="(blocker, index) in blockers" :key="index">
                    <div class="p-3 bg-red-50 border border-red-200 rounded-lg">
                        <p class="text-sm text-red-800" x-text="blocker"></p>
                    </div>
                </template>
            </div>
            
            <div class="flex flex-col gap-2">
                <template x-if="hasAssetBlocker">
                    <a 
                        href="#assets"
                        @click="showBlockers = false; scrollToSection('assets')"
                        class="inline-flex items-center justify-center px-4 py-2 text-sm font-medium text-white bg-blue-600 rounded-lg hover:bg-blue-700"
                    >
                        Ir para Assets
                    </a>
                </template>
                <template x-if="hasTaskBlocker">
                    <a 
                        href="#checklist"
                        @click="showBlockers = false; scrollToSection('checklist')"
                        class="inline-flex items-center justify-center px-4 py-2 text-sm font-medium text-white bg-blue-600 rounded-lg hover:bg-blue-700"
                    >
                        Ir para Checklist
                    </a>
                </template>
            </div>
            
            <div class="mt-4 flex justify-end">
                <button 
                    @click="showBlockers = false"
                    class="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-100"
                >
                    Fechar
                </button>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
    function statusTransition(episodeId, currentStatus) {
        return {
            episodeId: episodeId,
            currentStatus: currentStatus,
            transitions: [],
            loadingTransitions: false,
            showDropdown: false,
            showBlockers: false,
            blockers: [],
            
            async loadTransitions() {
                this.loadingTransitions = true;
                this.showDropdown = true;
                
                try {
                    const response = await window.apiFetch(`/api/episodes/${this.episodeId}/allowed-transitions`);
                    this.transitions = response.data || [];
                    
                    if (this.transitions.length === 0) {
                        window.showToast('Nenhuma transição disponível no momento', 'info');
                        this.showDropdown = false;
                    }
                } catch (error) {
                    console.error('Erro ao carregar transições:', error);
                    window.showToast('Erro ao carregar transições', 'error');
                    this.showDropdown = false;
                } finally {
                    this.loadingTransitions = false;
                }
            },
            
            confirmTransition(toStatus) {
                if (confirm(`Tem certeza que deseja mudar o status para "${this.getStatusLabel(toStatus)}"?`)) {
                    this.executeTransition(toStatus);
                }
            },
            
            async executeTransition(toStatus) {
                this.showDropdown = false;
                
                try {
                    const response = await window.apiFetch(`/api/episodes/${this.episodeId}/transition`, {
                        method: 'POST',
                        body: { to: toStatus },
                    });
                    
                    window.showToast('Status atualizado com sucesso!', 'success');
                    
                    // Recarrega a página para atualizar o badge
                    setTimeout(() => {
                        window.location.reload();
                    }, 500);
                } catch (error) {
                    console.error('Erro ao transicionar status:', error);
                    
                    // Se for erro 422 com bloqueios, exibe inline blockers
                    if (error.errors && error.errors.status) {
                        this.blockers = Array.isArray(error.errors.status) 
                            ? error.errors.status 
                            : [error.errors.status];
                        this.showBlockers = true;
                    } else if (error.message) {
                        // Se vier apenas message, exibe como blocker único
                        this.blockers = [error.message];
                        this.showBlockers = true;
                    } else {
                        window.showToast('Erro ao atualizar status', 'error');
                    }
                }
            },
            
            get hasAssetBlocker() {
                return this.blockers.some(b => 
                    b.toLowerCase().includes('asset') || 
                    b.toLowerCase().includes('link') ||
                    b.toLowerCase().includes('url')
                );
            },
            
            get hasTaskBlocker() {
                return this.blockers.some(b => 
                    b.toLowerCase().includes('tarefa') || 
                    b.toLowerCase().includes('checklist') ||
                    b.toLowerCase().includes('obrigatória')
                );
            },
            
            scrollToSection(sectionId) {
                const element = document.getElementById(sectionId) || document.querySelector(`[id*="${sectionId}"]`);
                if (element) {
                    element.scrollIntoView({ behavior: 'smooth', block: 'start' });
                    // Destaca visualmente
                    element.classList.add('ring-2', 'ring-blue-500', 'ring-offset-2');
                    setTimeout(() => {
                        element.classList.remove('ring-2', 'ring-blue-500', 'ring-offset-2');
                    }, 2000);
                }
            },
            
            getStatusLabel(status) {
                const labels = {
                    'pauta': 'Pauta',
                    'gravacao': 'Gravação',
                    'edicao': 'Edição',
                    'revisao': 'Revisão',
                    'agendado': 'Agendado',
                    'publicado': 'Publicado',
                };
                return labels[status] || status;
            }
        }
    }
</script>
@endpush
