@props(['episode'])

<div x-data="taskChecklist({{ $episode->id }})">
    <div class="space-y-2">
        <template x-if="loading">
            <div class="text-center py-4">
                <div class="inline-block animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div>
            </div>
        </template>

        <template x-if="!loading && tasks.length === 0">
            <p class="text-gray-500 text-sm">Nenhuma tarefa encontrada.</p>
        </template>

        <template x-for="task in tasks" :key="task.id">
            <div class="flex items-center gap-3 p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors">
                <input 
                    type="checkbox"
                    :checked="task.is_done"
                    @change="toggleTask(task.id, $event.target.checked)"
                    class="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 cursor-pointer"
                >
                <label 
                    :for="`task-${task.id}`"
                    class="flex-1 text-sm font-medium cursor-pointer"
                    :class="task.is_done ? 'text-gray-500 line-through' : 'text-gray-900'"
                    x-text="task.title"
                ></label>
                <template x-if="task.due_date">
                    <span class="text-xs text-gray-500" x-text="formatDate(task.due_date)"></span>
                </template>
            </div>
        </template>
    </div>
</div>

@push('scripts')
<script>
    function taskChecklist(episodeId) {
        return {
            episodeId: episodeId,
            tasks: [],
            loading: true,
            
            init() {
                this.loadTasks();
            },
            
            async loadTasks() {
                this.loading = true;
                try {
                    const response = await window.apiFetch(`/api/episodes/${this.episodeId}/tasks`);
                    this.tasks = response.data || [];
                } catch (error) {
                    console.error('Erro ao carregar tarefas:', error);
                    window.showToast('Erro ao carregar tarefas', 'error');
                } finally {
                    this.loading = false;
                }
            },
            
            async toggleTask(taskId, isDone) {
                // Optimistic UI - atualiza imediatamente
                const task = this.tasks.find(t => t.id === taskId);
                const previousState = task.is_done;
                task.is_done = isDone;
                
                try {
                    const response = await window.apiFetch(`/api/episodes/${this.episodeId}/tasks/${taskId}`, {
                        method: 'PATCH',
                        body: { is_done: isDone },
                    });
                    
                    // Atualiza com dados do servidor
                    const index = this.tasks.findIndex(t => t.id === taskId);
                    if (index > -1) {
                        this.tasks[index] = response.data;
                    }
                } catch (error) {
                    // Reverte em caso de erro
                    task.is_done = previousState;
                    console.error('Erro ao atualizar tarefa:', error);
                    window.showToast('Erro ao atualizar tarefa', 'error');
                }
            },
            
            formatDate(date) {
                if (!date) return '';
                return new Date(date).toLocaleDateString('pt-BR');
            }
        }
    }
</script>
@endpush
