# Documentação da API — Backend Gestor Editorial

**Versão:** 1.0  
**Base URL:** `http://localhost:8080`  
**Formato:** JSON  
**Autenticação:** Laravel Breeze (Session-based)

---

## 📋 Índice

1. [Visão Geral](#visão-geral)
2. [Autenticação](#autenticação)
3. [Formato de Resposta](#formato-de-resposta)
4. [Estruturas de Dados](#estruturas-de-dados)
5. [Endpoints](#endpoints)
   - [Episódios](#episódios)
   - [Convidados](#convidados)
   - [Temas](#temas)
   - [Tarefas de Episódio](#tarefas-de-episódio)
   - [Assets de Episódio](#assets-de-episódio)
   - [Calendário](#calendário)
   - [Relatórios](#relatórios)
6. [Fluxo de Status de Episódios](#fluxo-de-status-de-episódios)
7. [Validações e Erros](#validações-e-erros)

---

## Visão Geral

O backend é uma API RESTful construída com **Laravel 12**, que retorna dados em formato JSON. Todas as rotas (exceto autenticação) requerem autenticação via sessão.

### Características Principais

- **Autenticação:** Laravel Breeze (sessões)
- **Autorização:** Roles (admin/editor) via middleware
- **Paginação:** Laravel padrão (15 itens por página)
- **Relacionamentos:** Eager loading para evitar N+1 queries
- **Validação:** Form Requests com mensagens em português

---

## Autenticação

### Login

**POST** `/login`

```json
{
  "email": "admin@gestor-editorial.com",
  "password": "admin123"
}
```

**Resposta (200):**
```json
{
  "message": "Login realizado com sucesso"
}
```

### Logout

**POST** `/logout`

**Resposta (200):**
```json
{
  "message": "Logout realizado com sucesso"
}
```

### Usuário Autenticado

**GET** `/api/user` (se configurado) ou via sessão

**Resposta:**
```json
{
  "id": 1,
  "name": "Administrador",
  "email": "admin@gestor-editorial.com",
  "role": "admin"
}
```

**Roles disponíveis:**
- `admin` — Acesso total (pode deletar episódios, gerenciar usuários)
- `editor` — Pode criar e editar conteúdo

---

## Formato de Resposta

### Sucesso

```json
{
  "data": { ... },
  "message": "Operação realizada com sucesso"
}
```

### Erro de Validação (422)

```json
{
  "message": "The given data was invalid.",
  "errors": {
    "title": ["O campo título é obrigatório."],
    "status": ["O status selecionado é inválido."]
  }
}
```

### Erro de Autorização (403)

```json
{
  "message": "Unauthorized action."
}
```

### Erro de Recurso Não Encontrado (404)

```json
{
  "message": "No query results for model [App\\Models\\Episode] 1"
}
```

---

## Estruturas de Dados

### Episode (Episódio)

```json
{
  "id": 1,
  "title": "Inovação em Smart Cities",
  "slug": "inovacao-em-smart-cities",
  "record_date": "2026-01-15",
  "publish_date": "2026-01-20",
  "status": "publicado",
  "description": "Episódio sobre inovação...",
  "tags": "smart cities, inovação, tecnologia",
  "youtube_url": "https://youtube.com/watch?v=...",
  "spotify_url": "https://open.spotify.com/episode/...",
  "created_by": 1,
  "created_at": "2026-01-10T10:00:00.000000Z",
  "updated_at": "2026-01-20T15:30:00.000000Z",
  "creator": {
    "id": 1,
    "name": "Administrador",
    "email": "admin@gestor-editorial.com"
  },
  "guests": [
    {
      "id": 1,
      "name": "João Silva",
      "role_title": "Prefeito",
      "organization": "Prefeitura de Curitiba",
      "pivot": {
        "guest_role": "entrevistado"
      }
    }
  ],
  "themes": [
    {
      "id": 1,
      "name": "Smart Cities",
      "description": "Cidades inteligentes"
    }
  ],
  "tasks": [
    {
      "id": 1,
      "title": "Roteiro",
      "is_done": true,
      "due_date": "2026-01-12",
      "assigned_to": null
    }
  ],
  "assets": [
    {
      "id": 1,
      "asset_type": "audio",
      "label": "Áudio Final",
      "url": "https://drive.google.com/..."
    }
  ]
}
```

**Status possíveis:**
- `pauta` — Episódio em planejamento
- `gravacao` — Agendado para gravação
- `edicao` — Em edição
- `revisao` — Em revisão
- `agendado` — Agendado para publicação
- `publicado` — Publicado

### Guest (Convidado)

```json
{
  "id": 1,
  "name": "João Silva",
  "role_title": "Prefeito",
  "organization": "Prefeitura de Curitiba",
  "organization_id": 1,
  "bio_short": "Prefeito de Curitiba desde 2020...",
  "linkedin_url": "https://linkedin.com/in/joaosilva",
  "website_url": "https://joaosilva.com.br",
  "created_at": "2026-01-01T10:00:00.000000Z",
  "updated_at": "2026-01-01T10:00:00.000000Z",
  "organization": {
    "id": 1,
    "name": "Prefeitura de Curitiba"
  }
}
```

### Theme (Tema)

```json
{
  "id": 1,
  "name": "Smart Cities",
  "description": "Cidades inteligentes e tecnologia urbana",
  "created_at": "2026-01-01T10:00:00.000000Z",
  "updated_at": "2026-01-01T10:00:00.000000Z"
}
```

### EpisodeTask (Tarefa)

```json
{
  "id": 1,
  "episode_id": 1,
  "title": "Roteiro",
  "is_done": true,
  "is_critical": true,
  "due_date": "2026-01-12",
  "assigned_to": 2,
  "created_at": "2026-01-10T10:00:00.000000Z",
  "updated_at": "2026-01-12T14:00:00.000000Z",
  "assignee": {
    "id": 2,
    "name": "Editor Padrão",
    "email": "editor@gestor-editorial.com"
  }
}
```

**Tarefas padrão criadas automaticamente:**
- Roteiro (crítica)
- Confirmação do convidado (crítica)
- Gravação (crítica)
- Edição
- Cortes
- Thumb
- Publicação (crítica)
- Divulgação

### EpisodeAsset (Asset)

```json
{
  "id": 1,
  "episode_id": 1,
  "asset_type": "audio",
  "label": "Áudio Final",
  "url": "https://drive.google.com/file/d/...",
  "created_at": "2026-01-15T10:00:00.000000Z",
  "updated_at": "2026-01-15T10:00:00.000000Z"
}
```

**Tipos de asset sugeridos:**
- `audio` — Arquivo de áudio
- `video` — Arquivo de vídeo
- `script` — Roteiro
- `thumb` — Thumbnail
- `cuts` — Cortes/trechos
- `drive` — Link do Google Drive

### CalendarEvent (Evento do Calendário)

```json
{
  "id": "recording-1",
  "type": "recording",
  "date": "2026-01-15",
  "title": "Gravação: Inovação em Smart Cities",
  "color": "blue",
  "icon": "mic",
  "episode_id": 1,
  "episode": {
    "id": 1,
    "title": "Inovação em Smart Cities",
    "status": "gravacao",
    "slug": "inovacao-em-smart-cities"
  }
}
```

**Tipos de evento:**
- `recording` — Data de gravação (cor: blue, ícone: mic)
- `publication` — Data de publicação (cor: green, ícone: globe)
- `task` — Tarefa com prazo (cor: orange/gray, ícone: clipboard/check)

---

## Endpoints

### Episódios

#### Listar Episódios

**GET** `/episodes`

**Query Parameters:**
- `status` (opcional) — Filtrar por status
- `theme_id` (opcional) — Filtrar por tema
- `guest_id` (opcional) — Filtrar por convidado
- `sort_by` (opcional, padrão: `created_at`) — Campo para ordenação
- `sort_order` (opcional, padrão: `desc`) — `asc` ou `desc`
- `page` (opcional) — Página da paginação

**Resposta (200):**
```json
{
  "data": [
    {
      "id": 1,
      "title": "Inovação em Smart Cities",
      "slug": "inovacao-em-smart-cities",
      "status": "publicado",
      "record_date": "2026-01-15",
      "publish_date": "2026-01-20",
      "guests": [...],
      "themes": [...]
    }
  ],
  "links": {
    "first": "http://localhost:8080/episodes?page=1",
    "last": "http://localhost:8080/episodes?page=5",
    "prev": null,
    "next": "http://localhost:8080/episodes?page=2"
  },
  "meta": {
    "current_page": 1,
    "from": 1,
    "last_page": 5,
    "per_page": 15,
    "to": 15,
    "total": 72
  }
}
```

#### Criar Episódio

**POST** `/episodes`

**Body:**
```json
{
  "title": "Novo Episódio",
  "slug": "novo-episodio",
  "record_date": "2026-02-01",
  "publish_date": "2026-02-05",
  "status": "pauta",
  "description": "Descrição do episódio...",
  "tags": "tag1, tag2, tag3",
  "youtube_url": "https://youtube.com/watch?v=...",
  "spotify_url": null,
  "guests": [1, 2],
  "themes": [1, 3]
}
```

**Resposta (201):**
```json
{
  "data": {
    "id": 10,
    "title": "Novo Episódio",
    "status": "pauta",
    ...
  },
  "message": "Episódio criado com sucesso!"
}
```

**Nota:** Ao criar um episódio, as tarefas padrão são criadas automaticamente via `EpisodeObserver`.

#### Visualizar Episódio

**GET** `/episodes/{id}`

**Resposta (200):**
```json
{
  "data": {
    "id": 1,
    "title": "Inovação em Smart Cities",
    "slug": "inovacao-em-smart-cities",
    "status": "publicado",
    "guests": [...],
    "themes": [...],
    "tasks": [...],
    "assets": [...],
    "notes": [...],
    "metrics": [...]
  }
}
```

#### Atualizar Episódio

**PUT/PATCH** `/episodes/{id}`

**Body:** (mesmos campos do POST, todos opcionais)

**Resposta (200):**
```json
{
  "data": {
    "id": 1,
    "title": "Episódio Atualizado",
    ...
  },
  "message": "Episódio atualizado com sucesso!"
}
```

#### Deletar Episódio

**DELETE** `/episodes/{id}`

**Autorização:** Apenas `admin`

**Resposta (200):**
```json
{
  "message": "Episódio deletado com sucesso!"
}
```

#### Avançar Status

**POST** `/episodes/{id}/advance-status`

**Body:**
```json
{
  "next_status": "gravacao"
}
```

**Resposta (200):**
```json
{
  "success": true,
  "message": "Status alterado para 'gravacao' com sucesso.",
  "episode": {
    "id": 1,
    "status": "gravacao",
    ...
  }
}
```

**Resposta (422) — Erro de validação:**
```json
{
  "success": false,
  "errors": [
    "Transição de 'pauta' para 'edicao' não é permitida.",
    "Não é possível avançar: existem tarefas críticas não concluídas."
  ]
}
```

#### Transições Permitidas

**GET** `/episodes/{id}/allowed-transitions`

**Resposta (200):**
```json
{
  "allowed_transitions": ["gravacao"]
}
```

---

### Convidados

#### Listar Convidados

**GET** `/guests`

**Query Parameters:**
- `page` (opcional)

**Resposta (200):**
```json
{
  "data": [
    {
      "id": 1,
      "name": "João Silva",
      "role_title": "Prefeito",
      "organization": "Prefeitura de Curitiba",
      ...
    }
  ],
  "links": {...},
  "meta": {...}
}
```

#### Criar Convidado

**POST** `/guests`

**Body:**
```json
{
  "name": "Maria Santos",
  "role_title": "Secretária de Inovação",
  "organization": "Prefeitura de Curitiba",
  "organization_id": 1,
  "bio_short": "Secretária desde 2020...",
  "linkedin_url": "https://linkedin.com/in/mariasantos",
  "website_url": null
}
```

**Resposta (201):**
```json
{
  "data": {
    "id": 5,
    "name": "Maria Santos",
    ...
  },
  "message": "Convidado criado com sucesso!"
}
```

#### Visualizar Convidado

**GET** `/guests/{id}`

**Resposta (200):**
```json
{
  "data": {
    "id": 1,
    "name": "João Silva",
    "organization": {...},
    "episodes": [...]
  }
}
```

#### Atualizar Convidado

**PUT/PATCH** `/guests/{id}`

**Body:** (mesmos campos do POST)

**Resposta (200):**
```json
{
  "data": {...},
  "message": "Convidado atualizado com sucesso!"
}
```

#### Deletar Convidado

**DELETE** `/guests/{id}`

**Resposta (200):**
```json
{
  "message": "Convidado deletado com sucesso!"
}
```

---

### Temas

#### Listar Temas

**GET** `/themes`

**Resposta (200):**
```json
{
  "data": [
    {
      "id": 1,
      "name": "Smart Cities",
      "description": "Cidades inteligentes"
    }
  ],
  "links": {...},
  "meta": {...}
}
```

#### Criar Tema

**POST** `/themes`

**Body:**
```json
{
  "name": "GovTech",
  "description": "Tecnologia no setor público"
}
```

**Validação:** `name` deve ser único.

**Resposta (201):**
```json
{
  "data": {
    "id": 10,
    "name": "GovTech",
    ...
  },
  "message": "Tema criado com sucesso!"
}
```

#### Visualizar Tema

**GET** `/themes/{id}`

**Resposta (200):**
```json
{
  "data": {
    "id": 1,
    "name": "Smart Cities",
    "episodes": [...]
  }
}
```

#### Atualizar Tema

**PUT/PATCH** `/themes/{id}`

**Body:** (mesmos campos do POST)

**Resposta (200):**
```json
{
  "data": {...},
  "message": "Tema atualizado com sucesso!"
}
```

#### Deletar Tema

**DELETE** `/themes/{id}`

**Resposta (200):**
```json
{
  "message": "Tema deletado com sucesso!"
}
```

---

### Tarefas de Episódio

#### Listar Tarefas

**GET** `/episodes/{episode_id}/tasks`

**Resposta (200):**
```json
{
  "data": [
    {
      "id": 1,
      "title": "Roteiro",
      "is_done": true,
      "is_critical": true,
      "due_date": "2026-01-12",
      "assigned_to": 2
    }
  ]
}
```

#### Criar Tarefa

**POST** `/episodes/{episode_id}/tasks`

**Body:**
```json
{
  "title": "Revisão Final",
  "due_date": "2026-01-18",
  "assigned_to": 2,
  "is_critical": false
}
```

**Resposta (201):**
```json
{
  "data": {
    "id": 10,
    "title": "Revisão Final",
    "is_done": false,
    ...
  },
  "message": "Tarefa adicionada com sucesso!"
}
```

#### Atualizar Tarefa

**PATCH** `/episodes/{episode_id}/tasks/{task_id}`

**Body:**
```json
{
  "title": "Revisão Final Atualizada",
  "is_done": true,
  "due_date": "2026-01-18",
  "assigned_to": 2,
  "is_critical": false
}
```

**Resposta (200):**
```json
{
  "data": {...},
  "message": "Tarefa atualizada com sucesso!"
}
```

#### Deletar Tarefa

**DELETE** `/episodes/{episode_id}/tasks/{task_id}`

**Resposta (200):**
```json
{
  "message": "Tarefa removida com sucesso!"
}
```

---

### Assets de Episódio

#### Listar Assets

**GET** `/episodes/{episode_id}/assets`

**Resposta (200):**
```json
{
  "data": [
    {
      "id": 1,
      "asset_type": "audio",
      "label": "Áudio Final",
      "url": "https://drive.google.com/..."
    }
  ]
}
```

#### Criar Asset

**POST** `/episodes/{episode_id}/assets`

**Body:**
```json
{
  "asset_type": "video",
  "label": "Vídeo Completo",
  "url": "https://youtube.com/watch?v=..."
}
```

**Resposta (201):**
```json
{
  "data": {
    "id": 5,
    "asset_type": "video",
    ...
  },
  "message": "Asset adicionado com sucesso!"
}
```

#### Atualizar Asset

**PATCH** `/episodes/{episode_id}/assets/{asset_id}`

**Body:** (mesmos campos do POST)

**Resposta (200):**
```json
{
  "data": {...},
  "message": "Asset atualizado com sucesso!"
}
```

#### Deletar Asset

**DELETE** `/episodes/{episode_id}/assets/{asset_id}`

**Resposta (200):**
```json
{
  "message": "Asset removido com sucesso!"
}
```

---

### Calendário

#### Listar Eventos do Calendário

**GET** `/calendar`

**Query Parameters:**
- `view` (opcional, padrão: `month`) — `month` ou `week`
- `year` (obrigatório se `view=month`) — Ex: `2026`
- `month` (obrigatório se `view=month`) — Ex: `1` (janeiro)
- `start` (obrigatório se `view=week`) — Ex: `2026-01-01`
- `end` (obrigatório se `view=week`) — Ex: `2026-01-07`
- `event_types` (opcional) — Array: `["recording", "publication", "task"]`
- `statuses` (opcional) — Array: `["pauta", "gravacao"]`

**Exemplo — Mês:**
```
GET /calendar?view=month&year=2026&month=1
```

**Exemplo — Semana:**
```
GET /calendar?view=week&start=2026-01-01&end=2026-01-07
```

**Resposta (200):**
```json
{
  "events": {
    "2026-01-15": [
      {
        "id": "recording-1",
        "type": "recording",
        "date": "2026-01-15",
        "title": "Gravação: Inovação em Smart Cities",
        "color": "blue",
        "icon": "mic",
        "episode_id": 1,
        "episode": {
          "id": 1,
          "title": "Inovação em Smart Cities",
          "status": "gravacao",
          "slug": "inovacao-em-smart-cities"
        }
      }
    ],
    "2026-01-20": [
      {
        "id": "publication-1",
        "type": "publication",
        "date": "2026-01-20",
        "title": "Publicação: Inovação em Smart Cities",
        "color": "green",
        "icon": "globe",
        "episode_id": 1,
        "episode": {...}
      }
    ]
  },
  "view": "month"
}
```

---

### Relatórios

#### Top 10 Temas

**GET** `/reports/top-themes`

**Resposta (200):**
```json
[
  {
    "id": 1,
    "name": "Smart Cities",
    "description": "Cidades inteligentes",
    "episodes_count": 15
  },
  {
    "id": 2,
    "name": "GovTech",
    "description": "Tecnologia no setor público",
    "episodes_count": 12
  }
]
```

#### Top 10 Convidados

**GET** `/reports/top-guests`

**Resposta (200):**
```json
[
  {
    "id": 1,
    "name": "João Silva",
    "role_title": "Prefeito",
    "episodes_count": 8
  }
]
```

#### Episódios por Status

**GET** `/reports/episodes-by-status`

**Resposta (200):**
```json
[
  {
    "status": "publicado",
    "count": 45
  },
  {
    "status": "agendado",
    "count": 12
  },
  {
    "status": "pauta",
    "count": 8
  }
]
```

#### Tempo Médio de Publicação

**GET** `/reports/average-publish-time`

**Resposta (200):**
```json
{
  "average_days": 12.5,
  "message": "Tempo médio: 12.5 dias"
}
```

**Resposta (quando não há dados):**
```json
{
  "average_days": null,
  "message": "Sem dados suficientes"
}
```

---

## Fluxo de Status de Episódios

### Transições Permitidas

O sistema implementa uma máquina de estados para controlar o fluxo de status dos episódios:

```
pauta → gravacao → edicao → revisao → agendado → publicado
```

**Regras:**
1. Apenas transições diretas são permitidas (não pode pular etapas)
2. Não é possível voltar para um status anterior
3. Para avançar para `agendado`, é necessário:
   - Ter pelo menos uma URL (YouTube ou Spotify) preenchida
   - Ter todas as tarefas críticas concluídas

### Validações por Status

**Ao avançar para `agendado`:**
- ✅ Pelo menos uma URL (YouTube ou Spotify) deve estar preenchida
- ✅ Todas as tarefas críticas devem estar concluídas (`is_done = true`)

**Tarefas críticas padrão:**
- Roteiro
- Confirmação do convidado
- Gravação
- Publicação

### Exemplo de Uso

1. **Verificar transições permitidas:**
   ```
   GET /episodes/1/allowed-transitions
   → ["gravacao"]
   ```

2. **Avançar status:**
   ```
   POST /episodes/1/advance-status
   Body: { "next_status": "gravacao" }
   → Sucesso
   ```

3. **Tentar avançar sem validação:**
   ```
   POST /episodes/1/advance-status
   Body: { "next_status": "agendado" }
   → Erro: "Não é possível avançar: falta preencher youtube_url ou spotify_url."
   ```

---

## Validações e Erros

### Validações Comuns

#### Episódio

- `title`: obrigatório, string, máximo 255 caracteres
- `slug`: obrigatório, string, máximo 255 caracteres, único
- `status`: obrigatório, deve ser um dos: `pauta`, `gravacao`, `edicao`, `revisao`, `agendado`, `publicado`
- `record_date`: opcional, formato de data (`YYYY-MM-DD`)
- `publish_date`: opcional, formato de data, deve ser >= `record_date`
- `youtube_url`: opcional, URL válida, máximo 500 caracteres
- `spotify_url`: opcional, URL válida, máximo 500 caracteres
- `guests`: opcional, array de IDs existentes em `guests`
- `themes`: opcional, array de IDs existentes em `themes`

#### Convidado

- `name`: obrigatório, string, máximo 255 caracteres
- `role_title`: opcional, string, máximo 255 caracteres
- `organization_id`: opcional, deve existir em `organizations`
- `linkedin_url`: opcional, URL válida
- `website_url`: opcional, URL válida

#### Tema

- `name`: obrigatório, string, máximo 255 caracteres, único
- `description`: opcional, string

#### Tarefa

- `title`: obrigatório, string, máximo 255 caracteres
- `due_date`: opcional, formato de data
- `assigned_to`: opcional, deve existir em `users`
- `is_critical`: opcional, boolean

#### Asset

- `asset_type`: obrigatório, string, máximo 50 caracteres
- `label`: opcional, string, máximo 255 caracteres
- `url`: obrigatório, URL válida, máximo 2048 caracteres

### Códigos de Status HTTP

- `200` — Sucesso
- `201` — Criado com sucesso
- `204` — Sem conteúdo (deletado)
- `400` — Requisição inválida
- `401` — Não autenticado
- `403` — Não autorizado
- `404` — Recurso não encontrado
- `422` — Erro de validação
- `500` — Erro interno do servidor

### Tratamento de Erros no Frontend

**Exemplo de tratamento de erro de validação:**

```javascript
try {
  const response = await fetch('/episodes', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
    },
    body: JSON.stringify(data)
  });

  if (!response.ok) {
    const error = await response.json();
    if (error.errors) {
      // Exibir erros de validação
      Object.keys(error.errors).forEach(field => {
        console.error(`${field}: ${error.errors[field].join(', ')}`);
      });
    }
    throw new Error(error.message);
  }

  const result = await response.json();
  return result;
} catch (error) {
  console.error('Erro ao criar episódio:', error);
}
```

---

## Notas Importantes

1. **CSRF Protection:** Todas as requisições POST/PUT/PATCH/DELETE requerem token CSRF. No Laravel, isso é gerenciado automaticamente via sessão, mas em SPAs pode ser necessário configurar CORS e tokens.

2. **Eager Loading:** Os endpoints já incluem relacionamentos carregados (ex: `guests`, `themes`, `tasks`) para evitar múltiplas requisições.

3. **Paginação:** Listagens usam paginação padrão do Laravel (15 itens por página). Use os parâmetros `page` e os links em `links` e `meta` para navegação.

4. **Datas:** Todas as datas são retornadas no formato ISO 8601 (`YYYY-MM-DD` ou `YYYY-MM-DDTHH:mm:ss.sssZ`).

5. **Slug:** O slug do episódio é gerado automaticamente a partir do título, mas pode ser customizado. Deve ser único.

6. **Tarefas Padrão:** Ao criar um episódio, 8 tarefas padrão são criadas automaticamente. Elas podem ser editadas ou deletadas depois.

---

## Exemplos de Integração

### Criar Episódio Completo

```javascript
const episodeData = {
  title: "Inovação em Smart Cities",
  slug: "inovacao-em-smart-cities",
  record_date: "2026-02-01",
  publish_date: "2026-02-05",
  status: "pauta",
  description: "Episódio sobre inovação em cidades inteligentes...",
  tags: "smart cities, inovação, tecnologia",
  guests: [1, 2], // IDs dos convidados
  themes: [1, 3]  // IDs dos temas
};

const response = await fetch('/episodes', {
  method: 'POST',
  headers: {
    'Content-Type': 'application/json',
    'X-CSRF-TOKEN': csrfToken
  },
  body: JSON.stringify(episodeData)
});
```

### Avançar Status com Validação

```javascript
// 1. Verificar transições permitidas
const transitions = await fetch(`/episodes/${episodeId}/allowed-transitions`);
const { allowed_transitions } = await transitions.json();

if (allowed_transitions.length === 0) {
  alert('Não há transições disponíveis para este status.');
  return;
}

// 2. Tentar avançar
const response = await fetch(`/episodes/${episodeId}/advance-status`, {
  method: 'POST',
  headers: {
    'Content-Type': 'application/json',
    'X-CSRF-TOKEN': csrfToken
  },
  body: JSON.stringify({
    next_status: allowed_transitions[0]
  })
});

const result = await response.json();

if (!result.success) {
  // Exibir erros
  result.errors.forEach(error => alert(error));
} else {
  alert(result.message);
  // Atualizar interface
}
```

### Carregar Calendário Mensal

```javascript
const year = 2026;
const month = 1;

const response = await fetch(
  `/calendar?view=month&year=${year}&month=${month}`
);
const { events } = await response.json();

// events é um objeto onde a chave é a data (YYYY-MM-DD)
// e o valor é um array de eventos
Object.keys(events).forEach(date => {
  events[date].forEach(event => {
    console.log(`${date}: ${event.title} (${event.type})`);
  });
});
```

---

**Última atualização:** 2026-01-21  
**Versão da API:** 1.0  
**Contato:** Equipe de Backend
