Calendário não é feature estética — é instrumento de coordenação editorial. Se não pensar nele desde já, o sistema vira cadastro; se pensar bem, ele vira orquestrador de ritmo.

Vou estruturar a resposta em três camadas: conceito, modelo de dados e execução prática no MVP.

1️⃣ O que é o calendário nesse produto (conceito certo)

Aqui vai o ponto-chave:

O calendário não é uma entidade isolada.
Ele é uma vista temporal sobre o pipeline editorial.

Ou seja:

o episódio continua sendo o centro

o calendário apenas organiza eventos no tempo

não criamos “agenda paralela” que duplica informação

Isso evita:

inconsistência de datas

retrabalho

bugs de sincronização

2️⃣ Tipos de datas que fazem sentido (e por quê)
📅 Datas estruturais do episódio (core)

Essas já existem ou devem existir no modelo:

record_date → quando o episódio será/foi gravado

publish_date → quando o episódio será/foi publicado

Essas duas datas já permitem um calendário básico funcional.

🧭 Datas operacionais (derivadas)

Não precisam de coluna própria no início, mas entram no calendário como eventos calculados:

Início da edição (ex: record_date + 1)

Prazo de revisão

Data limite para assets (thumb, cortes)

Data de agendamento

Essas datas podem vir:

de tarefas com due_date

de regras do pipeline

📣 Datas de divulgação (futuro próximo)

Não entram no MVP inicial, mas já devem ser pensadas:

Post LinkedIn

Corte Instagram / Shorts

Newsletter

Essas podem virar:

episode_promotions (Sprint 4+)

ou tasks especializadas

3️⃣ Modelo de dados: o mínimo inteligente
Opção correta para agora (recomendada)

👉 Não criar tabela calendar_events no MVP.

Usar:

episodes.record_date

episodes.publish_date

episode_tasks.due_date

E gerar o calendário por query combinada.

Exemplo conceitual:

episódio com record_date → evento “Gravação”

episódio com publish_date → evento “Publicação”

task com due_date → evento “Tarefa”

Isso é:

simples

consistente

sem duplicação de dados

Evolução futura (quando fizer sentido)

Aí sim entra algo como:

calendar_events

id

entity_type (episode, task, promotion)

entity_id

event_type

start_date

end_date (opcional)

Mas não agora. Agora é execução.

4️⃣ Como o calendário aparece no MVP (UX prático)
📊 Visões de calendário que fazem sentido
🔹 Visão Mensal (principal)

mostra:

gravações

publicações

cores por tipo:

gravação → azul

publicação → verde

click leva para o episódio

🔹 Visão Semanal (opcional)

bom para operação

mostra tarefas com due_date

🔹 Filtro por status

só “agendados”

só “em edição”

só “publicados”

5️⃣ Queries base (conceito, não código final)

Eventos de gravação

episodes onde record_date IS NOT NULL

Eventos de publicação

episodes onde publish_date IS NOT NULL

Eventos de tarefas

episode_tasks onde due_date IS NOT NULL

Unifica tudo numa collection e renderiza no calendário.

6️⃣ Integração com o plano de execução (onde entra)

Eu colocaria assim no seu docs/plano_execucao.md:

Sprint 1.1 (ainda)

✔️ Episódio com datas estruturais

✔️ Tarefas com due_date

Sprint 2

🔹 View de Calendário (mensal)

🔹 Eventos derivados de episódios e tarefas

🔹 Filtros por tipo de evento

Sprint 4+

🔹 Eventos de divulgação

🔹 Export iCal / Google Calendar

🔹 SLA por etapa (tempo médio por fase)

7️⃣ Insight estratégico (importante mesmo)

Quando você implementa o calendário desse jeito:

o editor enxerga gargalos

o podcast ganha cadência

o sistema começa a coordenar tempo, não só dados

Isso é exatamente o tipo de coisa que diferencia:

"sistema interno"
de
"ferramenta editorial profissional"

---

## 8️⃣ Implementação Técnica no Backend Laravel

### Arquitetura: Service + Controller + DTO

**Princípio:** O calendário é uma **vista agregada**, não uma entidade. Tudo vem de queries sobre `episodes` e `episode_tasks`.

### 8.1 CalendarService (App\Services\CalendarService)

**Responsabilidade:** Agregar eventos de diferentes fontes e normalizar para uma estrutura única.

```php
namespace App\Services;

use App\Models\Episode;
use App\Models\EpisodeTask;
use Illuminate\Support\Collection;
use Carbon\Carbon;

class CalendarService
{
    /**
     * Busca eventos do calendário para um período
     * 
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @param array $filters ['event_types' => [], 'statuses' => []]
     * @return Collection<CalendarEvent>
     */
    public function getEvents(Carbon $startDate, Carbon $endDate, array $filters = []): Collection
    {
        $events = collect();
        
        // Eventos de gravação
        if (!$filters['event_types'] || in_array('recording', $filters['event_types'])) {
            $events = $events->merge($this->getRecordingEvents($startDate, $endDate, $filters));
        }
        
        // Eventos de publicação
        if (!$filters['event_types'] || in_array('publication', $filters['event_types'])) {
            $events = $events->merge($this->getPublicationEvents($startDate, $endDate, $filters));
        }
        
        // Eventos de tarefas
        if (!$filters['event_types'] || in_array('task', $filters['event_types'])) {
            $events = $events->merge($this->getTaskEvents($startDate, $endDate, $filters));
        }
        
        return $events->sortBy('date');
    }
    
    /**
     * Eventos de gravação (record_date)
     */
    private function getRecordingEvents(Carbon $start, Carbon $end, array $filters): Collection
    {
        $query = Episode::whereNotNull('record_date')
            ->whereBetween('record_date', [$start, $end])
            ->with(['guests', 'themes']);
            
        if (!empty($filters['statuses'])) {
            $query->whereIn('status', $filters['statuses']);
        }
        
        return $query->get()->map(function ($episode) {
            return new CalendarEvent([
                'id' => "recording-{$episode->id}",
                'type' => 'recording',
                'date' => $episode->record_date,
                'title' => "Gravação: {$episode->title}",
                'episode_id' => $episode->id,
                'episode' => $episode,
                'color' => 'blue',
                'icon' => 'mic',
            ]);
        });
    }
    
    /**
     * Eventos de publicação (publish_date)
     */
    private function getPublicationEvents(Carbon $start, Carbon $end, array $filters): Collection
    {
        $query = Episode::whereNotNull('publish_date')
            ->whereBetween('publish_date', [$start, $end])
            ->with(['guests', 'themes']);
            
        if (!empty($filters['statuses'])) {
            $query->whereIn('status', $filters['statuses']);
        }
        
        return $query->get()->map(function ($episode) {
            return new CalendarEvent([
                'id' => "publication-{$episode->id}",
                'type' => 'publication',
                'date' => $episode->publish_date,
                'title' => "Publicação: {$episode->title}",
                'episode_id' => $episode->id,
                'episode' => $episode,
                'color' => 'green',
                'icon' => 'broadcast',
            ]);
        });
    }
    
    /**
     * Eventos de tarefas (due_date)
     */
    private function getTaskEvents(Carbon $start, Carbon $end, array $filters): Collection
    {
        $query = EpisodeTask::whereNotNull('due_date')
            ->whereBetween('due_date', [$start, $end])
            ->with(['episode', 'assignee']);
            
        if (!empty($filters['statuses'])) {
            $query->whereHas('episode', function ($q) use ($filters) {
                $q->whereIn('status', $filters['statuses']);
            });
        }
        
        return $query->get()->map(function ($task) {
            return new CalendarEvent([
                'id' => "task-{$task->id}",
                'type' => 'task',
                'date' => $task->due_date,
                'title' => $task->title,
                'episode_id' => $task->episode_id,
                'episode' => $task->episode,
                'task' => $task,
                'color' => $task->is_done ? 'gray' : 'orange',
                'icon' => 'check-circle',
                'is_done' => $task->is_done,
            ]);
        });
    }
    
    /**
     * Eventos agrupados por mês (para visão mensal)
     */
    public function getEventsByMonth(int $year, int $month): Collection
    {
        $start = Carbon::create($year, $month, 1)->startOfMonth();
        $end = $start->copy()->endOfMonth();
        
        return $this->getEvents($start, $end)->groupBy(function ($event) {
            return $event->date->format('Y-m-d');
        });
    }
}
```

### 8.2 CalendarEvent DTO (App\DataTransferObjects\CalendarEvent)

**Responsabilidade:** Estrutura unificada para eventos do calendário.

```php
namespace App\DataTransferObjects;

use Carbon\Carbon;

class CalendarEvent
{
    public function __construct(
        public string $id,
        public string $type, // 'recording', 'publication', 'task'
        public Carbon $date,
        public string $title,
        public ?int $episode_id = null,
        public ?object $episode = null,
        public ?object $task = null,
        public string $color = 'gray',
        public string $icon = 'circle',
        public ?bool $is_done = null,
    ) {}
    
    public function toArray(): array
    {
        return [
            'id' => $this->id,
            'type' => $this->type,
            'date' => $this->date->format('Y-m-d'),
            'title' => $this->title,
            'episode_id' => $this->episode_id,
            'color' => $this->color,
            'icon' => $this->icon,
            'is_done' => $this->is_done,
            'episode' => $this->episode ? [
                'id' => $this->episode->id,
                'title' => $this->episode->title,
                'status' => $this->episode->status,
                'slug' => $this->episode->slug,
            ] : null,
        ];
    }
}
```

### 8.3 CalendarController

**Responsabilidade:** Expor endpoints para o calendário.

```php
namespace App\Http\Controllers;

use App\Services\CalendarService;
use Illuminate\Http\Request;
use Carbon\Carbon;

class CalendarController extends Controller
{
    public function __construct(
        private CalendarService $calendarService
    ) {}
    
    /**
     * GET /calendar?year=2026&month=1&view=month
     * GET /calendar?start=2026-01-01&end=2026-01-31&view=week
     */
    public function index(Request $request)
    {
        $view = $request->get('view', 'month'); // month, week
        $filters = [
            'event_types' => $request->get('event_types', []),
            'statuses' => $request->get('statuses', []),
        ];
        
        if ($view === 'month') {
            $year = $request->get('year', now()->year);
            $month = $request->get('month', now()->month);
            $events = $this->calendarService->getEventsByMonth($year, $month);
        } else {
            $start = Carbon::parse($request->get('start', now()->startOfWeek()));
            $end = Carbon::parse($request->get('end', now()->endOfWeek()));
            $events = $this->calendarService->getEvents($start, $end, $filters);
        }
        
        return response()->json([
            'events' => $events->map->toArray(),
            'view' => $view,
        ]);
    }
}
```

### 8.4 Rotas

```php
// routes/web.php
Route::middleware(['auth'])->group(function () {
    Route::get('/calendar', [CalendarController::class, 'index'])->name('calendar.index');
});
```

### 8.5 Otimizações e Índices

**Migrations de índices (já mencionado no plano_execucao.md):**

```php
// database/migrations/XXXX_add_calendar_indexes.php
Schema::table('episodes', function (Blueprint $table) {
    $table->index('record_date');
    $table->index('publish_date');
});

Schema::table('episode_tasks', function (Blueprint $table) {
    $table->index('due_date');
});
```

### 8.6 Integração com Frontend

**API Response (JSON):**

```json
{
  "events": {
    "2026-01-15": [
      {
        "id": "recording-1",
        "type": "recording",
        "date": "2026-01-15",
        "title": "Gravação: Episódio #42",
        "episode_id": 1,
        "color": "blue",
        "icon": "mic",
        "episode": {
          "id": 1,
          "title": "Episódio #42",
          "status": "gravacao",
          "slug": "episodio-42"
        }
      }
    ]
  },
  "view": "month"
}
```

### 8.7 Eventos Derivados (Futuro)

**Para calcular datas operacionais sem coluna própria:**

```php
// No CalendarService, adicionar método:
private function getDerivedEvents(Episode $episode): Collection
{
    $events = collect();
    
    // Início da edição (record_date + 1 dia)
    if ($episode->record_date) {
        $events->push(new CalendarEvent([
            'id' => "editing-start-{$episode->id}",
            'type' => 'derived',
            'date' => $episode->record_date->addDay(),
            'title' => "Início da edição: {$episode->title}",
            'color' => 'yellow',
        ]));
    }
    
    return $events;
}
```

### 8.8 Testes (Opcional, mas Recomendado)

```php
// tests/Feature/CalendarTest.php
public function test_calendar_returns_recording_events()
{
    $episode = Episode::factory()->create([
        'record_date' => '2026-01-15',
    ]);
    
    $service = new CalendarService();
    $events = $service->getEvents(
        Carbon::parse('2026-01-01'),
        Carbon::parse('2026-01-31')
    );
    
    $this->assertCount(1, $events);
    $this->assertEquals('recording', $events->first()->type);
}
```

---

## 9️⃣ Checklist de Implementação

### Sprint 2 — Calendário Básico

- [ ] Criar `CalendarService` com métodos de agregação
- [ ] Criar DTO `CalendarEvent`
- [ ] Criar `CalendarController` com endpoint JSON
- [ ] Adicionar rotas
- [ ] Criar migration de índices (`record_date`, `publish_date`, `due_date`)
- [ ] Testar queries (evitar N+1)
- [ ] View Blade básica (ou componente Vue/React)

### Sprint 4+ — Calendário Avançado

- [ ] Eventos derivados (datas calculadas)
- [ ] Export iCal / Google Calendar
- [ ] Filtros avançados (por convidado, tema)
- [ ] Visão semanal com tarefas
- [ ] SLA por etapa (tempo médio por fase)

---

## 🔟 Resumo Técnico

**Arquitetura:**
- ✅ Sem tabela `calendar_events` no MVP
- ✅ Service agrega de `episodes` e `episode_tasks`
- ✅ DTO unifica estrutura de eventos
- ✅ Controller expõe JSON para frontend
- ✅ Índices garantem performance

**Vantagens:**
- Simples (sem duplicação de dados)
- Consistente (fonte única de verdade)
- Performático (queries otimizadas com índices)
- Escalável (fácil adicionar novos tipos de eventos)
