# Especificação Técnica — Sistema de Curadoria

## 📋 Visão Geral

Este documento detalha a implementação técnica do sistema de curadoria, incluindo migrations, models, controllers e regras de negócio.

## 🗄️ Estrutura do Banco de Dados

### 1. Tabela `prospects`

```sql
CREATE TABLE prospects (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    
    -- Identificação
    type ENUM('person', 'company') NOT NULL,
    name VARCHAR(255) NOT NULL,
    email VARCHAR(255) NOT NULL,
    whatsapp VARCHAR(20) NULL,
    linkedin_url VARCHAR(255) NULL,
    website_url VARCHAR(255) NULL,
    
    -- Perfil
    role_title VARCHAR(255) NULL COMMENT 'Cargo/atuação (se person)',
    company_name VARCHAR(255) NULL COMMENT 'Empresa (se person)',
    organization_type ENUM('startup', 'empresa', 'governo', 'academia', 'osc', 'hub', 'outro') NULL,
    sector VARCHAR(100) NULL COMMENT 'Categoria/setor',
    location VARCHAR(255) NULL COMMENT 'Cidade/Estado',
    
    -- Sobre a Iniciativa (formulário público)
    what_they_do TEXT NULL COMMENT 'Em uma frase: o que fazem?',
    problem_solved TEXT NULL COMMENT 'Qual problema resolvem e para quem?',
    stage ENUM('ideacao', 'mvp', 'tracao', 'escala', 'consolidacao') NULL,
    
    -- Pauta e Valor (formulário público)
    expertise_topics TEXT NULL COMMENT '3 tópicos que domina',
    story_case TEXT NULL COMMENT 'História/caso real para compartilhar',
    ecosystem_request TEXT NULL COMMENT 'Pedido para o ecossistema',
    
    -- Disponibilidade (formulário público)
    preferred_format ENUM('presencial', 'online') NULL,
    preferred_time ENUM('manha', 'tarde', 'noite') NULL,
    availability_window TEXT NULL COMMENT 'Janela de disponibilidade',
    consent_given BOOLEAN DEFAULT FALSE COMMENT 'Autorização de contato',
    
    -- Curadoria
    origin ENUM('inbound_form', 'outbound_curated', 'referral', 'event', 'other') DEFAULT 'inbound_form',
    status VARCHAR(50) NOT NULL DEFAULT 'to_review' COMMENT 'to_review, approved, invited, in_conversation, scheduled, confirmed, recorded, published, follow_up, rejected, archived',
    priority TINYINT UNSIGNED NULL COMMENT '1-5',
    notes TEXT NULL COMMENT 'Justificativa editorial',
    fit_score TINYINT UNSIGNED NULL COMMENT '1-5',
    impact_potential TINYINT UNSIGNED NULL COMMENT '1-5',
    timing_score TINYINT UNSIGNED NULL COMMENT '1-5',
    total_score TINYINT UNSIGNED NULL COMMENT 'fit + impact + timing (3-15)',
    diversity_flags JSON NULL COMMENT 'Flags de diversidade',
    rejection_reason TEXT NULL COMMENT 'Motivo da rejeição (se rejected)',
    
    -- Relacionamentos
    created_by BIGINT UNSIGNED NULL COMMENT 'Usuário que cadastrou (NULL se formulário público)',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
    -- Relacionamento com Guest via guests.prospect_id (não precisa FK aqui)
    
    INDEX idx_status (status),
    INDEX idx_type (type),
    INDEX idx_origin (origin),
    INDEX idx_priority (priority),
    INDEX idx_total_score (total_score),
    INDEX idx_email (email)
);
```

### 2. Tabela `prospect_themes`

```sql
CREATE TABLE prospect_themes (
    prospect_id BIGINT UNSIGNED NOT NULL,
    theme_id BIGINT UNSIGNED NOT NULL,
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    
    PRIMARY KEY (prospect_id, theme_id),
    FOREIGN KEY (prospect_id) REFERENCES prospects(id) ON DELETE CASCADE,
    FOREIGN KEY (theme_id) REFERENCES themes(id) ON DELETE CASCADE
);
```

### 3. Alteração na tabela `guests`

Adicionar campo para rastrear origem:

```sql
ALTER TABLE guests 
ADD COLUMN prospect_id BIGINT UNSIGNED NULL AFTER organization_id,
ADD FOREIGN KEY (prospect_id) REFERENCES prospects(id) ON DELETE SET NULL;
```

## 📦 Models

### Model `Prospect`

```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Prospect extends Model
{
    protected $fillable = [
        // Identificação
        'type',
        'name',
        'email',
        'whatsapp',
        'linkedin_url',
        'website_url',
        
        // Perfil
        'role_title',
        'company_name',
        'organization_type',
        'sector',
        'location',
        
        // Sobre a Iniciativa
        'what_they_do',
        'problem_solved',
        'stage',
        
        // Pauta e Valor
        'expertise_topics',
        'story_case',
        'ecosystem_request',
        
        // Disponibilidade
        'preferred_format',
        'preferred_time',
        'availability_window',
        'consent_given',
        
        // Curadoria
        'origin',
        'status',
        'preferred_slots',
        'scheduled_at',
        'meeting_link',
        'priority',
        'notes',
        'fit_score',
        'impact_potential',
        'timing_score',
        'total_score',
        'diversity_flags',
        'rejection_reason',
        'created_by',
    ];

    protected $casts = [
        'diversity_flags' => 'array',
        'priority' => 'integer',
        'fit_score' => 'integer',
        'impact_potential' => 'integer',
        'timing_score' => 'integer',
        'total_score' => 'integer',
        'consent_given' => 'boolean',
    ];

    // Status disponíveis
    const STATUS_TO_REVIEW = 'to_review';
    const STATUS_APPROVED = 'approved';
    const STATUS_INVITED = 'invited';
    const STATUS_CONTACTED = 'contacted'; // Mantido para compatibilidade/legado
    const STATUS_IN_CONVERSATION = 'in_conversation';
    const STATUS_SCHEDULED = 'scheduled';
    const STATUS_CONFIRMED = 'confirmed';
    const STATUS_RECORDED = 'recorded';
    const STATUS_PUBLISHED = 'published';
    const STATUS_FOLLOW_UP = 'follow_up';
    const STATUS_REJECTED = 'rejected';
    const STATUS_ARCHIVED = 'archived';
    
    // Origens
    const ORIGIN_INBOUND_FORM = 'inbound_form';
    const ORIGIN_OUTBOUND_CURATED = 'outbound_curated';
    const ORIGIN_REFERRAL = 'referral';
    const ORIGIN_EVENT = 'event';
    const ORIGIN_OTHER = 'other';

    // Tipos
    const TYPE_PERSON = 'person';
    const TYPE_COMPANY = 'company';

    /**
     * Usuário que criou o prospect
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Temas relacionados ao prospect
     */
    public function themes(): BelongsToMany
    {
        return $this->belongsToMany(Theme::class, 'prospect_themes')
            ->withTimestamps();
    }

    /**
     * Guest criado a partir deste prospect
     * Relacionamento via guests.prospect_id (padrão único)
     */
    public function guest(): HasOne
    {
        return $this->hasOne(Guest::class, 'prospect_id');
    }

    /**
     * Calcula e atualiza o score total
     * Recalculado automaticamente no saving (via observer)
     */
    public function calculateTotalScore(): void
    {
        $this->total_score = ($this->fit_score ?? 0) 
            + ($this->impact_potential ?? 0) 
            + ($this->timing_score ?? 0);
    }
    
    /**
     * Recalcula score total antes de salvar
     * Registra mudanças de status automaticamente
     */
    protected static function booted(): void
    {
        static::saving(function (Prospect $prospect) {
            $prospect->calculateTotalScore();
        });
        
        static::updating(function (Prospect $prospect) {
            // Registrar mudança de status
            if ($prospect->isDirty('status')) {
                ProspectStatusLog::create([
                    'prospect_id' => $prospect->id,
                    'from_status' => $prospect->getOriginal('status'),
                    'to_status' => $prospect->status,
                    'user_id' => auth()->id(),
                ]);
            }
        });
    }

    /**
     * Verifica se pode ser convertido em Guest
     * Permite conversão a partir de scheduled (agilidade operacional)
     */
    public function canConvertToGuest(): bool
    {
        return in_array($this->status, [
            self::STATUS_SCHEDULED,
            self::STATUS_CONFIRMED,
        ]);
    }

    /**
     * Converte prospect em Guest
     */
    public function convertToGuest(): Guest
    {
        if (!$this->canConvertToGuest()) {
            throw new \Exception('Prospect não está confirmado para conversão');
        }

        // Verificar se já existe guest para este prospect
        if ($this->guest) {
            return $this->guest;
        }

        $guest = Guest::create([
            'name' => $this->name,
            'role_title' => $this->role_title,
            'organization' => $this->company_name,
            'bio_short' => $this->notes,
            'linkedin_url' => $this->linkedin_url,
            'website_url' => $this->website_url,
            'prospect_id' => $this->id, // Relacionamento único via guests.prospect_id
        ]);

        return $guest;
    }
}
```

### Atualização do Model `Guest`

```php
// Adicionar ao model Guest

protected $fillable = [
    // ... campos existentes
    'prospect_id',
];

/**
 * Prospect que originou este guest
 */
public function prospect(): BelongsTo
{
    return $this->belongsTo(Prospect::class);
}
```

## 🎮 Controllers

### `ProspectController`

```php
<?php

namespace App\Http\Controllers;

use App\Models\Prospect;
use App\Http\Requests\StoreProspectRequest;
use App\Http\Requests\UpdateProspectRequest;
use Illuminate\Http\Request;

class ProspectController extends Controller
{
    public function index(Request $request)
    {
        $query = Prospect::with(['creator', 'themes', 'guest'])
            ->orderBy('priority', 'desc')
            ->orderBy('total_score', 'desc');

        // Filtros
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }
        if ($request->has('origin')) {
            $query->where('origin', $request->origin);
        }
        if ($request->has('theme_id')) {
            $query->whereHas('themes', function($q) use ($request) {
                $q->where('themes.id', $request->theme_id);
            });
        }

        $prospects = $query->paginate(15);

        return view('prospects.index', compact('prospects'));
    }

    public function create()
    {
        $themes = Theme::all();
        return view('prospects.create', compact('themes'));
    }

    public function store(StoreProspectRequest $request)
    {
        $prospect = Prospect::create([
            ...$request->validated(),
            'created_by' => auth()->id(),
        ]);

        if ($request->has('themes')) {
            $prospect->themes()->attach($request->themes);
        }

        $prospect->calculateTotalScore();

        return redirect()->route('prospects.index')
            ->with('success', 'Prospect cadastrado com sucesso');
    }

    public function show(Prospect $prospect)
    {
        $prospect->load(['creator', 'themes', 'guest']);
        return view('prospects.show', compact('prospect'));
    }

    public function edit(Prospect $prospect)
    {
        $themes = Theme::all();
        $prospect->load('themes');
        return view('prospects.edit', compact('prospect', 'themes'));
    }

    public function update(UpdateProspectRequest $request, Prospect $prospect)
    {
        $prospect->update($request->validated());

        if ($request->has('themes')) {
            $prospect->themes()->sync($request->themes);
        }

        $prospect->calculateTotalScore();

        return redirect()->route('prospects.show', $prospect)
            ->with('success', 'Prospect atualizado com sucesso');
    }

    public function approve(Prospect $prospect)
    {
        $prospect->update(['status' => Prospect::STATUS_APPROVED]);
        return redirect()->back()->with('success', 'Prospect aprovado');
    }

    public function reject(Request $request, Prospect $prospect)
    {
        $request->validate([
            'rejection_reason' => 'required|string|min:10',
        ]);

        $prospect->update([
            'status' => Prospect::STATUS_REJECTED,
            'rejection_reason' => $request->rejection_reason,
        ]);

        return redirect()->back()->with('success', 'Prospect rejeitado');
    }

    public function updateStatus(Request $request, Prospect $prospect)
    {
        $request->validate([
            'status' => 'required|in:' . implode(',', [
                Prospect::STATUS_APPROVED,
                Prospect::STATUS_CONTACTED,
                Prospect::STATUS_IN_CONVERSATION,
                Prospect::STATUS_SCHEDULED,
                Prospect::STATUS_CONFIRMED,
            ]),
        ]);

        $prospect->update(['status' => $request->status]);

        return redirect()->back()->with('success', 'Status atualizado');
    }

    public function convertToGuest(Prospect $prospect)
    {
        try {
            $guest = $prospect->convertToGuest();
            
            // Sincronizar temas
            if ($prospect->themes->isNotEmpty()) {
                // Temas serão associados quando o episódio for criado
            }

            return redirect()->route('guests.show', $guest)
                ->with('success', 'Prospect convertido em convidado com sucesso');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
}
```

## 📝 Form Requests

### `StoreProspectRequest`

```php
<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StoreProspectRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'type' => 'required|in:person,company',
            'name' => 'required|string|max:255',
            'role_title' => 'nullable|string|max:255',
            'company_name' => 'nullable|string|max:255',
            'sector' => 'nullable|string|max:100',
            'location' => 'nullable|string|max:255',
            'origin' => 'required|in:inbound_form,outbound_curated,referral,event,other',
            'priority' => 'nullable|integer|min:1|max:5',
            'notes' => 'nullable|string',
            'linkedin_url' => 'nullable|url|max:255',
            'website_url' => 'nullable|url|max:255',
            'fit_score' => 'nullable|integer|min:1|max:5',
            'impact_potential' => 'nullable|integer|min:1|max:5',
            'timing_score' => 'nullable|integer|min:1|max:5',
            'diversity_flags' => 'nullable|array',
            'themes' => 'nullable|array',
            'themes.*' => 'exists:themes,id',
        ];
    }
}
```

## 🎨 Views (Estrutura)

### Kanban View (`prospects/index.blade.php`)

```blade
<div class="kanban-board">
    @foreach(['to_review', 'approved', 'contacted', 'in_conversation', 'scheduled', 'confirmed'] as $status)
        <div class="kanban-column" data-status="{{ $status }}">
            <h3>{{ ucfirst(str_replace('_', ' ', $status)) }}</h3>
            @foreach($prospects->where('status', $status) as $prospect)
                @include('prospects.partials.card', ['prospect' => $prospect])
            @endforeach
        </div>
    @endforeach
</div>
```

## 🔄 Migrations

### Migration: `create_prospects_table`

```php
<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('prospects', function (Blueprint $table) {
            $table->id();
            $table->enum('type', ['person', 'company']);
            $table->string('name');
            $table->string('role_title')->nullable();
            $table->string('company_name')->nullable();
            $table->string('sector', 100)->nullable();
            $table->string('location')->nullable();
            $table->enum('origin', ['inbound_form', 'outbound_curated', 'referral', 'event', 'other'])->default('inbound_form');
            $table->string('status', 50)->default('to_review');
            $table->tinyInteger('priority')->nullable();
            $table->text('notes')->nullable();
            $table->string('linkedin_url')->nullable();
            $table->string('website_url')->nullable();
            $table->tinyInteger('fit_score')->nullable();
            $table->tinyInteger('impact_potential')->nullable();
            $table->tinyInteger('timing_score')->nullable();
            $table->tinyInteger('total_score')->nullable();
            $table->json('diversity_flags')->nullable();
            $table->text('rejection_reason')->nullable();
            
            // Relacionamentos
            // Relacionamento com Guest via guests.prospect_id (não precisa FK aqui)
            $table->foreignId('created_by')->nullable()->constrained('users')->nullOnDelete();
            $table->timestamps();

            $table->index('status');
            $table->index('type');
            $table->index('origin');
            $table->index('priority');
            $table->index('total_score');
            $table->index('email');
            
            // Campos de agendamento
            $table->text('preferred_slots')->nullable()->after('availability_window');
            $table->dateTime('scheduled_at')->nullable()->after('preferred_slots');
            $table->string('meeting_link')->nullable()->after('scheduled_at');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('prospects');
    }
};
```

### Migration: `create_prospect_themes_table`

```php
<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('prospect_themes', function (Blueprint $table) {
            $table->foreignId('prospect_id')->constrained('prospects')->cascadeOnDelete();
            $table->foreignId('theme_id')->constrained('themes')->cascadeOnDelete();
            $table->primary(['prospect_id', 'theme_id']);
            $table->timestamps();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('prospect_themes');
    }
};
```

### Migration: `add_prospect_id_to_guests_table`

```php
<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::table('guests', function (Blueprint $table) {
            $table->foreignId('prospect_id')->nullable()->after('organization_id')
                ->constrained('prospects')->nullOnDelete();
        });
    }

    public function down(): void
    {
        Schema::table('guests', function (Blueprint $table) {
            $table->dropForeign(['prospect_id']);
            $table->dropColumn('prospect_id');
        });
    }
};
```

## 🛣️ Rotas

```php
// routes/web.php

// Formulário público (sem autenticação)
Route::get('manifestar-interesse', [PublicProspectController::class, 'show'])
    ->name('public.prospects.show');
Route::post('manifestar-interesse', [PublicProspectController::class, 'store'])
    ->name('public.prospects.store');

// Rotas autenticadas
Route::middleware(['auth'])->group(function () {
    Route::resource('prospects', ProspectController::class);
    
    Route::post('prospects/{prospect}/approve', [ProspectController::class, 'approve'])
        ->name('prospects.approve');
    Route::post('prospects/{prospect}/reject', [ProspectController::class, 'reject'])
        ->name('prospects.reject');
    Route::post('prospects/{prospect}/status', [ProspectController::class, 'updateStatus'])
        ->name('prospects.update-status');
    Route::post('prospects/{prospect}/convert-to-guest', [ProspectController::class, 'convertToGuest'])
        ->name('prospects.convert-to-guest');
});
```

## 📊 Relatórios e Insights

### Queries úteis

```php
// Prospects aprovados não contatados
Prospect::where('status', 'approved')
    ->where('updated_at', '<', now()->subDays(7))
    ->count();

// Taxa de conversão por origem
DB::table('prospects')
    ->select('origin', DB::raw('COUNT(*) as total'), 
             DB::raw('SUM(CASE WHEN EXISTS (
                 SELECT 1 FROM guests WHERE guests.prospect_id = prospects.id
             ) THEN 1 ELSE 0 END) as converted'))
    ->groupBy('origin')
    ->get();

// Tempo médio no funil
DB::table('prospects')
    ->whereExists(function($query) {
        $query->select(DB::raw(1))
              ->from('guests')
              ->whereColumn('guests.prospect_id', 'prospects.id');
    })
    ->selectRaw('AVG(DATEDIFF(updated_at, created_at)) as avg_days')
    ->first();
```

## 📝 Formulário Público de Captação

### Controller: `PublicProspectController`

```php
<?php

namespace App\Http\Controllers;

use App\Models\Prospect;
use App\Models\Theme;
use App\Http\Requests\StorePublicProspectRequest;
use Illuminate\Http\Request;

class PublicProspectController extends Controller
{
    public function show()
    {
        $themes = Theme::orderBy('name')->get();
        return view('public.prospects.form', compact('themes'));
    }

    public function store(StorePublicProspectRequest $request)
    {
        // Criar prospect automaticamente
        $prospect = Prospect::create([
            'type' => 'person', // Formulário sempre cria pessoa
            'name' => $request->name,
            'email' => $request->email,
            'whatsapp' => $request->whatsapp,
            'linkedin_url' => $request->linkedin_url,
            'role_title' => $request->role_title,
            'company_name' => $request->company_name,
            'organization_type' => $request->organization_type,
            'location' => $request->location,
            'what_they_do' => $request->what_they_do,
            'problem_solved' => $request->problem_solved,
            'stage' => $request->stage,
            'expertise_topics' => $request->expertise_topics,
            'story_case' => $request->story_case,
            'ecosystem_request' => $request->ecosystem_request,
            'preferred_format' => $request->preferred_format,
            'preferred_time' => $request->preferred_time,
            'availability_window' => $request->availability_window,
            'consent_given' => $request->has('consent_given'),
            'origin' => Prospect::ORIGIN_INBOUND_FORM,
            'status' => Prospect::STATUS_TO_REVIEW,
            'created_by' => null, // Cadastro público
        ]);

        // Associar temas
        if ($request->has('themes')) {
            $prospect->themes()->attach($request->themes);
        }

        // Enviar email de confirmação (opcional)
        // Mail::to($prospect->email)->send(new ProspectConfirmation($prospect));

        return redirect()->route('public.prospects.thanks')
            ->with('success', 'Sua manifestação de interesse foi registrada com sucesso!');
    }

    public function thanks()
    {
        return view('public.prospects.thanks');
    }
}
```

### Form Request: `StorePublicProspectRequest`

```php
<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StorePublicProspectRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            // Bloco A - Identificação
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'whatsapp' => 'nullable|string|max:20',
            'linkedin_url' => 'nullable|url|max:255',
            
            // Bloco B - Perfil
            'organization_type' => 'required|in:startup,empresa,governo,academia,osc,hub,outro',
            'company_name' => 'required|string|max:255',
            'role_title' => 'required|string|max:255',
            'location' => 'required|string|max:255',
            
            // Bloco C - Sobre a Iniciativa
            'what_they_do' => 'required|string|max:500',
            'problem_solved' => 'required|string|max:500',
            'stage' => 'required|in:ideacao,mvp,tracao,escala,consolidacao',
            
            // Bloco D - Pauta e Valor
            'themes' => 'required|array|min:1',
            'themes.*' => 'exists:themes,id',
            'expertise_topics' => 'required|string|max:500',
            'story_case' => 'required|string|max:500',
            'ecosystem_request' => 'nullable|string|max:500',
            
            // Bloco E - Disponibilidade
            'preferred_format' => 'required|in:presencial,online',
            'preferred_time' => 'required|in:manha,tarde,noite',
            'availability_window' => 'nullable|string|max:500',
            'consent_given' => 'required|accepted',
        ];
    }

    public function messages(): array
    {
        return [
            'name.required' => 'O nome completo é obrigatório.',
            'email.required' => 'O e-mail é obrigatório.',
            'email.email' => 'Por favor, informe um e-mail válido.',
            'organization_type.required' => 'Selecione o tipo de organização.',
            'company_name.required' => 'O nome da organização é obrigatório.',
            'role_title.required' => 'O cargo/função é obrigatório.',
            'location.required' => 'A cidade/estado é obrigatória.',
            'what_they_do.required' => 'Descreva o que vocês fazem.',
            'problem_solved.required' => 'Descreva qual problema vocês resolvem.',
            'stage.required' => 'Selecione o estágio atual.',
            'themes.required' => 'Selecione pelo menos um tema.',
            'expertise_topics.required' => 'Informe os 3 tópicos que você domina.',
            'story_case.required' => 'Compartilhe uma história/caso real.',
            'preferred_format.required' => 'Selecione o formato preferido.',
            'preferred_time.required' => 'Selecione o melhor período.',
            'consent_given.required' => 'Você precisa autorizar o contato.',
        ];
    }
}
```

### View: Formulário Público (`resources/views/public/prospects/form.blade.php`)

```blade
@extends('layouts.public')

@section('title', 'Manifestar Interesse - Podcast Vale do Pinhão')

@section('content')
<div class="container mx-auto px-4 py-8">
    <div class="max-w-3xl mx-auto">
        <h1 class="text-3xl font-bold mb-4">Manifestar Interesse em Participar</h1>
        
        <div class="bg-blue-50 border-l-4 border-blue-500 p-4 mb-6">
            <p class="text-sm text-blue-700">
                <strong>Importante:</strong> Este formulário é uma <strong>manifestação de interesse</strong>, 
                não uma inscrição direta. A seleção é curada, alinhada com nossa pauta e agenda editorial. 
                Todos que preencherem entram no nosso radar e podem ser chamados em ciclos futuros.
            </p>
        </div>

        <form action="{{ route('public.prospects.store') }}" method="POST" class="space-y-6">
            @csrf

            <!-- Bloco A: Identificação -->
            <div class="bg-white p-6 rounded-lg shadow">
                <h2 class="text-xl font-semibold mb-4">Identificação</h2>
                <!-- Campos do formulário -->
            </div>

            <!-- Bloco B: Perfil -->
            <div class="bg-white p-6 rounded-lg shadow">
                <h2 class="text-xl font-semibold mb-4">Perfil</h2>
                <!-- Campos do formulário -->
            </div>

            <!-- Bloco C: Sobre a Iniciativa -->
            <div class="bg-white p-6 rounded-lg shadow">
                <h2 class="text-xl font-semibold mb-4">Sobre a Iniciativa</h2>
                <!-- Campos do formulário -->
            </div>

            <!-- Bloco D: Pauta e Valor -->
            <div class="bg-white p-6 rounded-lg shadow">
                <h2 class="text-xl font-semibold mb-4">Pauta e Valor da Entrevista</h2>
                <!-- Campos do formulário -->
            </div>

            <!-- Bloco E: Disponibilidade -->
            <div class="bg-white p-6 rounded-lg shadow">
                <h2 class="text-xl font-semibold mb-4">Disponibilidade e Logística</h2>
                <!-- Campos do formulário -->
            </div>

            <!-- Bloco F: Consentimento -->
            <div class="bg-white p-6 rounded-lg shadow">
                <div class="flex items-start">
                    <input type="checkbox" name="consent_given" id="consent_given" 
                           class="mt-1 mr-2" required>
                    <label for="consent_given" class="text-sm">
                        Entendo que este formulário é uma manifestação de interesse e que a 
                        participação depende de curadoria editorial e disponibilidade de agenda.
                    </label>
                </div>
                @error('consent_given')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <button type="submit" class="w-full bg-blue-600 text-white py-3 rounded-lg 
                   hover:bg-blue-700 font-semibold">
                Enviar Manifestação de Interesse
            </button>
        </form>
    </div>
</div>
@endsection
```

### View: Página de Agradecimento (`resources/views/public/prospects/thanks.blade.php`)

```blade
@extends('layouts.public')

@section('title', 'Obrigado - Podcast Vale do Pinhão')

@section('content')
<div class="container mx-auto px-4 py-8">
    <div class="max-w-2xl mx-auto text-center">
        <div class="bg-green-50 border-l-4 border-green-500 p-6 rounded-lg mb-6">
            <h1 class="text-3xl font-bold text-green-700 mb-4">Obrigado!</h1>
            <p class="text-lg text-gray-700">
                Sua manifestação de interesse foi registrada com sucesso.
            </p>
        </div>
        
        <div class="bg-white p-6 rounded-lg shadow">
            <p class="text-gray-600 mb-4">
                Nossa equipe de curadoria vai analisar sua manifestação e entrar em contato 
                caso haja alinhamento com nossa pauta editorial e disponibilidade de agenda.
            </p>
            <p class="text-sm text-gray-500">
                Todos que preencherem o formulário entram no nosso radar e podem ser 
                chamados em ciclos futuros.
            </p>
        </div>
    </div>
</div>
@endsection
```

### Migration Atualizada: `create_prospects_table`

```php
<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('prospects', function (Blueprint $table) {
            $table->id();
            
            // Identificação
            $table->enum('type', ['person', 'company']);
            $table->string('name');
            $table->string('email');
            $table->string('whatsapp', 20)->nullable();
            $table->string('linkedin_url')->nullable();
            $table->string('website_url')->nullable();
            
            // Perfil
            $table->string('role_title')->nullable();
            $table->string('company_name')->nullable();
            $table->enum('organization_type', ['startup', 'empresa', 'governo', 'academia', 'osc', 'hub', 'outro'])->nullable();
            $table->string('sector', 100)->nullable();
            $table->string('location')->nullable();
            
            // Sobre a Iniciativa
            $table->text('what_they_do')->nullable();
            $table->text('problem_solved')->nullable();
            $table->enum('stage', ['ideacao', 'mvp', 'tracao', 'escala', 'consolidacao'])->nullable();
            
            // Pauta e Valor
            $table->text('expertise_topics')->nullable();
            $table->text('story_case')->nullable();
            $table->text('ecosystem_request')->nullable();
            
            // Disponibilidade
            $table->enum('preferred_format', ['presencial', 'online'])->nullable();
            $table->enum('preferred_time', ['manha', 'tarde', 'noite'])->nullable();
            $table->text('availability_window')->nullable();
            $table->boolean('consent_given')->default(false);
            
            // Curadoria
            $table->enum('origin', ['inbound_form', 'outbound_curated', 'referral', 'event', 'other'])->default('inbound_form');
            $table->string('status', 50)->default('to_review')
                ->comment('to_review, approved, invited, in_conversation, scheduled, confirmed, recorded, published, follow_up, rejected, archived');
            $table->tinyInteger('priority')->nullable();
            $table->text('notes')->nullable();
            $table->tinyInteger('fit_score')->nullable();
            $table->tinyInteger('impact_potential')->nullable();
            $table->tinyInteger('timing_score')->nullable();
            $table->tinyInteger('total_score')->nullable();
            $table->json('diversity_flags')->nullable();
            $table->text('rejection_reason')->nullable();
            
            // Relacionamentos
            // Relacionamento com Guest via guests.prospect_id (não precisa FK aqui)
            $table->foreignId('created_by')->nullable()->constrained('users')->nullOnDelete();
            $table->timestamps();

            $table->index('status');
            $table->index('type');
            $table->index('origin');
            $table->index('priority');
            $table->index('total_score');
            $table->index('email');
            
            // Campos de agendamento
            $table->text('preferred_slots')->nullable()->after('availability_window');
            $table->dateTime('scheduled_at')->nullable()->after('preferred_slots');
            $table->string('meeting_link')->nullable()->after('scheduled_at');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('prospects');
    }
};
```

## 🤖 Assistente do Curador - Implementação

### Model `ProspectStatusLog`

```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ProspectStatusLog extends Model
{
    protected $fillable = [
        'prospect_id',
        'from_status',
        'to_status',
        'user_id',
        'notes',
    ];

    protected $casts = [
        'created_at' => 'datetime',
    ];

    public function prospect(): BelongsTo
    {
        return $this->belongsTo(Prospect::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }
}
```

### Model `ProspectContact`

```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ProspectContact extends Model
{
    protected $fillable = [
        'prospect_id',
        'channel',
        'direction',
        'message_subject',
        'message_body',
        'sent_at',
        'status',
        'created_by',
    ];

    protected $casts = [
        'sent_at' => 'datetime',
    ];

    const CHANNEL_EMAIL = 'email';
    const CHANNEL_WHATSAPP = 'whatsapp';
    const CHANNEL_LINKEDIN = 'linkedin';
    const CHANNEL_PHONE = 'phone';

    const DIRECTION_OUTBOUND = 'outbound';
    const DIRECTION_INBOUND = 'inbound';

    const STATUS_SENT = 'sent';
    const STATUS_REPLIED = 'replied';
    const STATUS_BOUNCED = 'bounced';
    const STATUS_FAILED = 'failed';

    public function prospect(): BelongsTo
    {
        return $this->belongsTo(Prospect::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }
}
```

### Model `ProspectIntakeToken`

```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;
use Carbon\Carbon;

class ProspectIntakeToken extends Model
{
    protected $fillable = [
        'prospect_id',
        'token',
        'expires_at',
        'completed_at',
        'created_by',
    ];

    protected $casts = [
        'expires_at' => 'datetime',
        'completed_at' => 'datetime',
    ];

    public function prospect(): BelongsTo
    {
        return $this->belongsTo(Prospect::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public static function generate(Prospect $prospect, int $daysValid = 30): self
    {
        return self::create([
            'prospect_id' => $prospect->id,
            'token' => Str::random(64),
            'expires_at' => Carbon::now()->addDays($daysValid),
            'created_by' => auth()->id(),
        ]);
    }

    public function isValid(): bool
    {
        return !$this->isExpired() && !$this->isCompleted();
    }

    public function isExpired(): bool
    {
        return $this->expires_at->isPast();
    }

    public function isCompleted(): bool
    {
        return $this->completed_at !== null;
    }

    public function getUrl(): string
    {
        return route('public.guest-intake.show', ['token' => $this->token]);
    }
}
```

### Model `GuestIntakeSubmission`

```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class GuestIntakeSubmission extends Model
{
    protected $fillable = [
        'prospect_id',
        'guest_id',
        'bio',
        'official_links',
        'photo_url',
        'favorite_themes',
        'availability',
        'image_voice_consent',
        'submitted_at',
    ];

    protected $casts = [
        'official_links' => 'array',
        'favorite_themes' => 'array',
        'image_voice_consent' => 'boolean',
        'submitted_at' => 'datetime',
    ];

    public function prospect(): BelongsTo
    {
        return $this->belongsTo(Prospect::class);
    }

    public function guest(): BelongsTo
    {
        return $this->belongsTo(Guest::class);
    }
}
```

### Atualização do Model `Prospect` - Relacionamentos

```php
// Adicionar ao model Prospect

/**
 * Contatos com este prospect
 */
public function contacts(): HasMany
{
    return $this->hasMany(ProspectContact::class);
}

/**
 * Tokens de ficha do convidado
 */
public function intakeTokens(): HasMany
{
    return $this->hasMany(ProspectIntakeToken::class);
}

/**
 * Submissão de ficha do convidado
 */
public function intakeSubmission(): HasOne
{
    return $this->hasOne(GuestIntakeSubmission::class);
}

/**
 * Logs de mudança de status
 */
public function statusLogs(): HasMany
{
    return $this->hasMany(ProspectStatusLog::class);
}

/**
 * Timeline de eventos (contatos + mudanças de status)
 */
public function getTimeline(): Collection
{
    $timeline = collect();
    
    // Evento de criação
    $timeline->push([
        'type' => 'created',
        'date' => $this->created_at,
        'label' => 'Criado',
        'description' => "Origem: {$this->origin}",
    ]);
    
    // Contatos
    foreach ($this->contacts as $contact) {
        $timeline->push([
            'type' => 'contact',
            'date' => $contact->sent_at ?? $contact->created_at,
            'label' => "Contato via {$contact->channel}",
            'description' => ucfirst($contact->direction),
            'contact' => $contact,
        ]);
    }
    
    // Mudanças de status (histórico completo)
    foreach ($this->statusLogs as $log) {
        $timeline->push([
            'type' => 'status_change',
            'date' => $log->created_at,
            'label' => "Status: {$log->from_status} → {$log->to_status}",
            'description' => $log->user?->name ?? 'Sistema',
            'log' => $log,
        ]);
    }
    
    return $timeline->sortBy('date');
}
```

### Controller: Métodos do Assistente do Curador

```php
// Adicionar ao ProspectController

/**
 * Gerar convite (template + personalização)
 */
public function generateInvite(Request $request, Prospect $prospect)
{
    $request->validate([
        'template' => 'required|in:startup,governo,academia,empresa,default',
    ]);

    // Buscar template do config (versionado)
    $templates = config('prospect_invites.templates', []);
    $template = $templates[$request->template] ?? $templates['default'];
    
    // Preencher variáveis
    $variables = [
        'name' => $prospect->name,
        'organization' => $prospect->company_name ?? $prospect->name,
        'theme' => $prospect->themes->first()?->name ?? 'Tema a definir',
        'host' => config('prospect_invites.default_host', 'Alan e Américo'),
        'format' => $prospect->preferred_format === 'presencial' ? 'presencial em Curitiba' : 'online',
    ];

    $emailBody = $this->replaceVariables($template['email'] ?? '', $variables);
    $whatsappBody = $this->replaceVariables($template['whatsapp'] ?? '', $variables);
    $linkedinBody = $this->replaceVariables($template['linkedin'] ?? '', $variables);

    return response()->json([
        'email' => $emailBody,
        'whatsapp' => $whatsappBody,
        'linkedin' => $linkedinBody,
        'variables' => $variables,
    ]);
}

/**
 * Substitui variáveis no template
 */
private function replaceVariables(string $template, array $variables): string
{
    $result = $template;
    foreach ($variables as $key => $value) {
        $result = str_replace('{' . $key . '}', $value, $result);
    }
    return $result;
}

/**
 * Registrar envio de contato
 */
public function logContact(Request $request, Prospect $prospect)
{
    $request->validate([
        'channel' => 'required|in:email,whatsapp,linkedin,phone',
        'direction' => 'required|in:outbound,inbound',
        'message_subject' => 'nullable|string|max:255',
        'message_body' => 'nullable|string',
        'sent_at' => 'nullable|date',
    ]);

    $contact = ProspectContact::create([
        'prospect_id' => $prospect->id,
        'channel' => $request->channel,
        'direction' => $request->direction,
        'message_subject' => $request->message_subject,
        'message_body' => $request->message_body,
        'sent_at' => $request->sent_at ?? now(),
        'status' => ProspectContact::STATUS_SENT,
        'created_by' => auth()->id(),
    ]);

    // Atualizar status do prospect se necessário
    if ($prospect->status === Prospect::STATUS_APPROVED && $request->direction === 'outbound') {
        $prospect->update(['status' => Prospect::STATUS_INVITED]);
    }

    return redirect()->back()->with('success', 'Contato registrado com sucesso');
}

/**
 * Gerar link de ficha do convidado
 */
public function generateIntakeLink(Prospect $prospect)
{
    $token = ProspectIntakeToken::generate($prospect);
    
    return response()->json([
        'url' => $token->getUrl(),
        'token' => $token->token,
        'expires_at' => $token->expires_at->format('d/m/Y H:i'),
    ]);
}

/**
 * Agendar entrevista
 */
public function scheduleInterview(Request $request, Prospect $prospect)
{
    $request->validate([
        'scheduled_at' => 'required|date|after:now',
        'meeting_link' => 'nullable|url',
        'preferred_slots' => 'nullable|string',
    ]);

    $prospect->update([
        'scheduled_at' => $request->scheduled_at,
        'meeting_link' => $request->meeting_link,
        'preferred_slots' => $request->preferred_slots,
        'status' => Prospect::STATUS_SCHEDULED,
    ]);

    // Criar episódio em Pauta
    $episode = Episode::create([
        'title' => "Episódio com {$prospect->name}",
        'slug' => \Str::slug("episodio-com-{$prospect->name}"),
        'status' => 'pauta',
        'record_date' => $request->scheduled_at,
        'created_by' => auth()->id(),
    ]);

    // Converter prospect em guest se ainda não foi
    if (!$prospect->guest) {
        $guest = $prospect->convertToGuest();
    } else {
        $guest = $prospect->guest;
    }

    // Vincular guest ao episódio
    $episode->guests()->attach($guest->id);

    // Vincular temas
    if ($prospect->themes->isNotEmpty()) {
        $episode->themes()->sync($prospect->themes->pluck('id'));
    }

    // Criar tasks padrão
    $this->createDefaultTasks($episode);

    return redirect()->route('episodes.show', $episode)
        ->with('success', 'Entrevista agendada e episódio criado');
}

private function createDefaultTasks(Episode $episode): void
{
    $defaultTasks = [
        'Roteiro',
        'Confirmação do convidado',
        'Gravação',
        'Edição',
        'Revisão',
        'Thumb',
        'Publicação',
    ];

    foreach ($defaultTasks as $taskTitle) {
        EpisodeTask::create([
            'episode_id' => $episode->id,
            'title' => $taskTitle,
            'is_done' => false,
        ]);
    }
}
```

### Rotas Adicionais

```php
// routes/web.php

Route::middleware(['auth'])->group(function () {
    // Assistente do Curador
    Route::post('prospects/{prospect}/generate-invite', [ProspectController::class, 'generateInvite'])
        ->name('prospects.generate-invite');
    Route::post('prospects/{prospect}/log-contact', [ProspectController::class, 'logContact'])
        ->name('prospects.log-contact');
    Route::post('prospects/{prospect}/generate-intake-link', [ProspectController::class, 'generateIntakeLink'])
        ->name('prospects.generate-intake-link');
    Route::post('prospects/{prospect}/schedule-interview', [ProspectController::class, 'scheduleInterview'])
        ->name('prospects.schedule-interview');
});

// Formulário público de ficha do convidado
Route::get('ficha-convidado/{token}', [PublicGuestIntakeController::class, 'show'])
    ->name('public.guest-intake.show');
Route::post('ficha-convidado/{token}', [PublicGuestIntakeController::class, 'store'])
    ->name('public.guest-intake.store');
```

## 📋 Configuração: Templates de Convite

### Arquivo: `config/prospect_invites.php`

```php
<?php

return [
    'default_host' => 'Alan e Américo',
    
    'templates' => [
        'startup' => [
            'email' => <<<'EMAIL'
Olá {name},

Somos {host} e produzimos o podcast Vale do Pinhão, que explora o ecossistema de inovação de Curitiba.

Gostaríamos de convidá-lo(a) para participar de um episódio sobre {theme}.

Você representa {organization} e acreditamos que sua experiência seria muito valiosa para nossa audiência.

O formato seria {format}, e podemos agendar conforme sua disponibilidade.

Aguardo seu retorno para alinharmos os detalhes.

Abraços,
{host}
EMAIL,
            'whatsapp' => <<<'WHATSAPP'
Olá {name}! 👋

Somos {host} do podcast Vale do Pinhão.

Gostaríamos de convidá-lo(a) para um episódio sobre {theme}.

Você representa {organization} e sua experiência seria muito valiosa!

Formato: {format}

Podemos agendar conforme sua disponibilidade. O que acha?

Aguardo seu retorno! 🎙️
WHATSAPP,
            'linkedin' => <<<'LINKEDIN'
Olá {name},

Somos {host} do podcast Vale do Pinhão.

Gostaríamos de convidá-lo(a) para participar de um episódio sobre {theme}.

Você representa {organization} e acreditamos que sua experiência seria muito valiosa.

Formato: {format}

Podemos agendar conforme sua disponibilidade.

Aguardo seu retorno!
LINKEDIN,
        ],
        
        'governo' => [
            'email' => <<<'EMAIL'
Prezado(a) {name},

Somos {host} e produzimos o podcast Vale do Pinhão, que explora o ecossistema de inovação de Curitiba.

Gostaríamos de convidá-lo(a) para participar de um episódio sobre {theme}, representando {organization}.

Acreditamos que sua experiência no setor público seria muito valiosa para nossa audiência.

O formato seria {format}, e podemos agendar conforme sua disponibilidade.

Aguardo seu retorno para alinharmos os detalhes.

Atenciosamente,
{host}
EMAIL,
            'whatsapp' => <<<'WHATSAPP'
Olá {name}! 👋

Somos {host} do podcast Vale do Pinhão.

Gostaríamos de convidá-lo(a) para um episódio sobre {theme}, representando {organization}.

Sua experiência no setor público seria muito valiosa!

Formato: {format}

Podemos agendar conforme sua disponibilidade.

Aguardo seu retorno! 🎙️
WHATSAPP,
            'linkedin' => <<<'LINKEDIN'
Prezado(a) {name},

Somos {host} do podcast Vale do Pinhão.

Gostaríamos de convidá-lo(a) para participar de um episódio sobre {theme}, representando {organization}.

Sua experiência no setor público seria muito valiosa.

Formato: {format}

Podemos agendar conforme sua disponibilidade.

Aguardo seu retorno!
LINKEDIN,
        ],
        
        'default' => [
            'email' => <<<'EMAIL'
Olá {name},

Somos {host} e produzimos o podcast Vale do Pinhão.

Gostaríamos de convidá-lo(a) para participar de um episódio sobre {theme}.

Você representa {organization} e acreditamos que sua experiência seria muito valiosa.

O formato seria {format}, e podemos agendar conforme sua disponibilidade.

Aguardo seu retorno para alinharmos os detalhes.

Abraços,
{host}
EMAIL,
            'whatsapp' => <<<'WHATSAPP'
Olá {name}! 👋

Somos {host} do podcast Vale do Pinhão.

Gostaríamos de convidá-lo(a) para um episódio sobre {theme}.

Você representa {organization} e sua experiência seria muito valiosa!

Formato: {format}

Podemos agendar conforme sua disponibilidade.

Aguardo seu retorno! 🎙️
WHATSAPP,
            'linkedin' => <<<'LINKEDIN'
Olá {name},

Somos {host} do podcast Vale do Pinhão.

Gostaríamos de convidá-lo(a) para participar de um episódio sobre {theme}.

Você representa {organization} e acreditamos que sua experiência seria muito valiosa.

Formato: {format}

Podemos agendar conforme sua disponibilidade.

Aguardo seu retorno!
LINKEDIN,
        ],
    ],
];
```

### Migration: `create_prospect_status_logs_table`

```php
<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('prospect_status_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('prospect_id')->constrained('prospects')->cascadeOnDelete();
            $table->string('from_status', 50)->nullable();
            $table->string('to_status', 50);
            $table->foreignId('user_id')->nullable()->constrained('users')->nullOnDelete();
            $table->text('notes')->nullable();
            $table->timestamp('created_at');

            $table->index('prospect_id');
            $table->index('to_status');
            $table->index('created_at');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('prospect_status_logs');
    }
};
```

---

## ✅ Resumo das Correções Aplicadas

### 1. ✅ Padronização `origin` (removido `source`)
- Todas as referências a `source` foram substituídas por `origin`
- Filtros, validações e índices atualizados
- Migration consolidada (sem duplicação)

### 2. ✅ Relação Prospect → Guest corrigida
- Removido `prospects.guest_id` (duplicidade)
- Mantido apenas `guests.prospect_id` (padrão único)
- Relacionamento corrigido: `hasOne(Guest::class, 'prospect_id')`

### 3. ✅ Status constants completos
- Adicionado `STATUS_CONTACTED` (compatibilidade)
- Pipeline expandido documentado

### 4. ✅ Regra de conversão ajustada
- `canConvertToGuest()` permite conversão a partir de `scheduled`
- Compatível com `scheduleInterview()` que converte ao agendar

### 5. ✅ Migration consolidada
- Uma única migration final com todos os campos
- Sem duplicação ou conflitos

### 6. ✅ Melhorias implementadas
- **total_score automático**: Recalculado no `saving` (observer)
- **Logs de status**: Tabela `prospect_status_logs` + registro automático
- **Templates versionados**: Config `prospect_invites.php` para fácil edição

---

**Próximos passos**: Implementar migrations, models, controllers e views conforme especificado acima.
