

---

## `docs/frontend/api_contract.md`

````md
# API Contract — Frontend Integration Guide
Podcast Editorial Manager

Este documento define como o frontend deve consumir a API backend
(Laravel 12 + Breeze + Session Auth), incluindo padrões de request,
response, erros e boas práticas de integração.

---

## 1) Princípios Gerais

- Autenticação baseada em **session** (Laravel Breeze)
- Frontend e backend compartilham o mesmo domínio
- Comunicação via **JSON**
- Frontend **não replica regras de negócio**
- Backend é a fonte da verdade (status, transições, validações)

---

## 2) Headers Padrão

### Requests mutáveis (POST, PUT, PATCH, DELETE)
Obrigatório enviar:

- `Accept: application/json`
- `X-CSRF-TOKEN: <csrf_token>`

O CSRF token deve ser obtido:
- via meta tag (`<meta name="csrf-token">`)
- ou variável global injetada no layout Blade

---

## 3) Padrão de Respostas

### 3.1 Sucesso (200 / 201)

```json
{
  "data": { ... }
}
````

Ou, em listagens:

```json
{
  "data": [ ... ],
  "meta": {
    "current_page": 1,
    "per_page": 15,
    "total": 120
  }
}
```

Frontend deve:

* usar apenas `data`
* ignorar campos extras não documentados

---

### 3.2 Erro de Validação (422)

Formato padrão Laravel:

```json
{
  "message": "The given data was invalid.",
  "errors": {
    "title": ["O título é obrigatório."],
    "publish_date": ["Data inválida."]
  }
}
```

**Tratamento no frontend:**

* mapear `errors[field]` para mensagens por campo
* exibir toast genérico opcional (“Corrija os erros do formulário”)
* não tratar como erro fatal

---

### 3.3 Não Autorizado / Sessão expirada (401 / 419)

Possíveis causas:

* usuário não autenticado
* CSRF inválido ou sessão expirada

**Tratamento no frontend:**

* redirecionar para `/login`
* opcional: toast “Sessão expirada, faça login novamente”

---

### 3.4 Proibido (403)

```json
{
  "message": "This action is unauthorized."
}
```

**Tratamento no frontend:**

* exibir toast “Você não tem permissão para esta ação”
* não redirecionar automaticamente

---

### 3.5 Erro de Negócio / Bloqueio de Regra (422 específico)

Usado principalmente em:

* transição de status
* ações condicionais

Exemplo:

```json
{
  "message": "Transição não permitida",
  "errors": {
    "status": [
      "Não é possível publicar um episódio sem links de publicação.",
      "Existem tarefas obrigatórias pendentes."
    ]
  }
}
```

**Tratamento no frontend:**

* exibir mensagens claramente (lista)
* não tentar corrigir automaticamente
* orientar o usuário

---

## 4) Endpoints Principais (Visão Frontend)

### 4.1 Episódios

#### Listar

`GET /api/episodes`

Query params:

* `status`
* `theme_id`
* `guest_id`
* `q`

Response:

```json
{
  "data": [
    {
      "id": 1,
      "title": "Episódio X",
      "status": "edicao",
      "publish_date": "2026-02-10",
      "guests": [ ... ],
      "themes": [ ... ]
    }
  ]
}
```

---

#### Detalhe

`GET /api/episodes/{id}`

Inclui:

* guests
* themes
* tasks
* assets

---

#### Criar / Atualizar

`POST /api/episodes`
`PUT /api/episodes/{id}`

Payload esperado:

```json
{
  "title": "Título",
  "record_date": "2026-01-20",
  "publish_date": "2026-02-10",
  "description": "...",
  "tags": "govtech, inovação",
  "guest_ids": [1, 2],
  "theme_ids": [3, 4]
}
```

---

### 4.2 Tasks (Checklist)

#### Toggle task

`PATCH /api/episodes/{episode_id}/tasks/{task_id}`

Payload:

```json
{
  "is_done": true
}
```

Response:

```json
{
  "data": {
    "id": 10,
    "is_done": true
  }
}
```

Frontend deve:

* usar optimistic UI
* reverter se erro

---

### 4.3 Assets

#### Criar

`POST /api/episodes/{episode_id}/assets`

Payload:

```json
{
  "asset_type": "video",
  "label": "Versão final",
  "url": "https://..."
}
```

---

#### Deletar

`DELETE /api/episodes/{episode_id}/assets/{asset_id}`

Response:

* 204 No Content

---

### 4.4 Status & Máquina de Estados

#### Transições permitidas

`GET /api/episodes/{id}/allowed-transitions`

Response:

```json
{
  "data": ["revisao", "agendado"]
}
```

Frontend:

* só exibe opções retornadas
* não tenta inferir regras

---

#### Executar transição

`POST /api/episodes/{id}/transition`

Payload:

```json
{
  "to": "agendado"
}
```

---

### 4.5 Calendário

`GET /api/calendar?month=2026-01`

Response:

```json
{
  "events": {
    "2026-01-20": [
      {
        "type": "recording",
        "label": "Gravação — Episódio X",
        "episode_id": 1
      }
    ]
  }
}
```

Frontend:

* agrupar por dia
* renderizar cores por `type`
* clique leva ao episódio

---

### 4.6 Relatórios

#### Temas mais recorrentes

`GET /api/reports/top-themes`

```json
{
  "data": [
    { "theme": "GovTech", "count": 12 }
  ]
}
```

---

#### Tempo médio gravação → publicação

`GET /api/reports/avg-publish-time`

```json
{
  "data": {
    "average_days": 14
  }
}
```

---

## 5) Convenção de Helper Frontend

### apiFetch(url, options)

Responsabilidades:

* incluir headers padrão
* parse JSON
* lançar exceção customizada por status
* retornar `{ data }`

Não usar:

* axios (no MVP)
* fetch direto espalhado pelo código

---

## 6) O que o Frontend NÃO deve fazer

* ❌ Inferir transições de status
* ❌ Reimplementar regras de validação
* ❌ Assumir estrutura fixa de responses não documentadas
* ❌ Silenciar erros de negócio

---

## 7) Objetivo do Contrato

Este contrato existe para garantir:

* desenvolvimento paralelo FE/BE
* previsibilidade
* menos bugs de integração
* onboarding rápido de novos devs

Se algo não estiver coberto aqui, a regra é:
👉 **backend decide, frontend se adapta**

```


