<?php
/**
 * Script PHP para criar ZIP otimizado para produção
 * Exclui arquivos desnecessários (vendor, node_modules, docker, testes, etc.)
 * 
 * Uso: php criar-zip-producao.php [nome-do-arquivo.zip]
 */

$outputFile = $argv[1] ?? 'gestor-editorial-producao.zip';
$rootDir = dirname(__DIR__);
$tempDir = sys_get_temp_dir() . '/gestor-editorial-temp-' . uniqid();
$outputPath = $rootDir . '/' . $outputFile;

// Cores
$colors = [
    'reset' => "\033[0m",
    'red' => "\033[31m",
    'green' => "\033[32m",
    'yellow' => "\033[33m",
    'blue' => "\033[34m",
    'cyan' => "\033[36m",
];

function colorize($text, $color = 'reset') {
    global $colors;
    if (php_sapi_name() !== 'cli' || !function_exists('posix_isatty') || !@posix_isatty(STDOUT)) {
        return $text;
    }
    return $colors[$color] . $text . $colors['reset'];
}

echo colorize("═══════════════════════════════════════════════════════════\n", 'cyan');
echo colorize("  CRIANDO ZIP OTIMIZADO PARA PRODUÇÃO\n", 'green');
echo colorize("═══════════════════════════════════════════════════════════\n\n", 'cyan');

// Verificar extensão ZIP
if (!extension_loaded('zip')) {
    echo colorize("ERRO: Extensão ZIP do PHP não está habilitada!\n", 'red');
    echo "Instale a extensão: sudo apt-get install php-zip (Linux) ou habilite no php.ini\n";
    exit(1);
}

// Padrões a excluir
$excludePatterns = [
    // Vendor e dependências
    'backend/vendor',
    'backend/node_modules',
    
    // Docker
    'docker',
    
    // Testes
    'backend/tests',
    
    // Git
    '.git',
    'backend/.git',
    
    // Arquivos de desenvolvimento
    'backend/.idea',
    'backend/.vscode',
    'backend/.fleet',
    'backend/.nova',
    'backend/.phpactor.json',
    'backend/.phpunit.result.cache',
    'backend/.phpunit.cache',
    'backend/Homestead.json',
    'backend/Homestead.yaml',
    
    // Cursor/IDE
    '.cursor',
    'backend/.cursor',
    
    // Database SQLite local
    'backend/database/database.sqlite',
    
    // Scripts de desenvolvimento
    'backend/init-laravel.ps1',
];

// Função para verificar se deve excluir
function shouldExclude($path, $excludePatterns) {
    $relativePath = str_replace('\\', '/', $path);
    foreach ($excludePatterns as $pattern) {
        if (strpos($relativePath, $pattern) !== false) {
            return true;
        }
    }
    
    // Excluir arquivos específicos
    $filename = basename($path);
    $excludeFiles = ['.env', '.env.backup', '.env.production', 'Thumbs.db', '.DS_Store'];
    if (in_array($filename, $excludeFiles)) {
        return true;
    }
    
    // Excluir logs
    if (preg_match('/\.log$/', $filename)) {
        return true;
    }
    
    // Excluir cache do storage (mas manter estrutura)
    if (preg_match('#backend/storage/framework/(cache|sessions|views)/#', $relativePath)) {
        return true;
    }
    
    return false;
}

// Função recursiva para adicionar arquivos ao ZIP
function addToZip($zip, $dir, $baseDir, $excludePatterns) {
    $files = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );
    
    foreach ($files as $file) {
        $filePath = $file->getRealPath();
        $relativePath = str_replace($baseDir . '/', '', str_replace('\\', '/', $filePath));
        
        if (shouldExclude($relativePath, $excludePatterns)) {
            continue;
        }
        
        if ($file->isDir()) {
            // Criar diretório no ZIP
            $zip->addEmptyDir($relativePath);
        } else {
            // Adicionar arquivo ao ZIP
            $zip->addFile($filePath, $relativePath);
        }
    }
}

echo colorize("[1] Criando diretório temporário...\n", 'cyan');
if (!mkdir($tempDir, 0755, true)) {
    echo colorize("ERRO: Não foi possível criar diretório temporário!\n", 'red');
    exit(1);
}

$projectDir = $tempDir . '/gestor-editorial';
mkdir($projectDir, 0755, true);

echo colorize("[2] Copiando arquivos necessários...\n", 'cyan');

// Copiar backend
echo colorize("   Copiando backend...\n", 'blue');
copyDirectory($rootDir . '/backend', $projectDir . '/backend', $excludePatterns);

// Limpar storage (manter estrutura)
echo colorize("   Limpando storage...\n", 'blue');
$storageDirs = [
    'storage/framework/cache',
    'storage/framework/sessions',
    'storage/framework/views',
    'storage/logs'
];
foreach ($storageDirs as $dir) {
    $fullPath = $projectDir . '/backend/' . $dir;
    if (is_dir($fullPath)) {
        $files = glob($fullPath . '/*');
        foreach ($files as $file) {
            if (is_file($file)) {
                unlink($file);
            }
        }
    }
}

// Copiar instalador
echo colorize("   Copiando instalador...\n", 'blue');
copyDirectory($rootDir . '/instalador', $projectDir . '/instalador', []);

// Copiar docs
echo colorize("   Copiando documentação...\n", 'blue');
if (is_dir($rootDir . '/docs')) {
    copyDirectory($rootDir . '/docs', $projectDir . '/docs', []);
}

// Copiar readme
if (file_exists($rootDir . '/readme.md')) {
    copy($rootDir . '/readme.md', $projectDir . '/readme.md');
}

echo colorize("[3] Criando arquivo ZIP...\n", 'cyan');

// Remover ZIP anterior se existir
if (file_exists($outputPath)) {
    unlink($outputPath);
}

// Criar ZIP
$zip = new ZipArchive();
if ($zip->open($outputPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
    echo colorize("ERRO: Não foi possível criar arquivo ZIP!\n", 'red');
    exit(1);
}

addToZip($zip, $projectDir, $projectDir, []);

$zip->close();

echo colorize("[4] Limpando diretório temporário...\n", 'cyan');
removeDirectory($tempDir);

// Calcular tamanho
$zipSize = filesize($outputPath);
$zipSizeMB = round($zipSize / (1024 * 1024), 2);

echo "\n";
echo colorize("═══════════════════════════════════════════════════════════\n", 'cyan');
echo colorize("  ZIP CRIADO COM SUCESSO!\n", 'green');
echo colorize("═══════════════════════════════════════════════════════════\n\n", 'cyan');

echo colorize("Arquivo: $outputPath\n", 'yellow');
echo colorize("Tamanho: $zipSizeMB MB\n", 'yellow');
echo colorize("\nO que foi excluído:\n", 'blue');
echo "  - vendor/ (será reinstalado via composer)\n";
echo "  - node_modules/ (será reinstalado via npm)\n";
echo "  - docker/ (não necessário em produção)\n";
echo "  - tests/ (testes não são necessários)\n";
echo "  - .git/ (controle de versão)\n";
echo "  - Logs e cache do storage/\n";
echo "  - Arquivos de desenvolvimento\n";
echo colorize("\nPróximos passos:\n", 'blue');
echo "  1. Faça upload do arquivo ZIP no filemanager\n";
echo "  2. Extraia o ZIP no servidor\n";
echo "  3. Execute: php instalador/instalar.php\n";
echo "\n";

// Funções auxiliares
function copyDirectory($src, $dst, $excludePatterns) {
    $dir = opendir($src);
    @mkdir($dst, 0755, true);
    
    while (($file = readdir($dir)) !== false) {
        if ($file != '.' && $file != '..') {
            $srcFile = $src . '/' . $file;
            $dstFile = $dst . '/' . $file;
            $relativePath = str_replace(dirname(__DIR__) . '/', '', str_replace('\\', '/', $srcFile));
            
            if (shouldExclude($relativePath, $excludePatterns)) {
                continue;
            }
            
            if (is_dir($srcFile)) {
                copyDirectory($srcFile, $dstFile, $excludePatterns);
            } else {
                copy($srcFile, $dstFile);
            }
        }
    }
    closedir($dir);
}

function removeDirectory($dir) {
    if (!is_dir($dir)) {
        return;
    }
    $files = array_diff(scandir($dir), ['.', '..']);
    foreach ($files as $file) {
        $path = $dir . '/' . $file;
        is_dir($path) ? removeDirectory($path) : unlink($path);
    }
    rmdir($dir);
}
