# Script para criar ZIP otimizado para produção
# Exclui arquivos desnecessários (vendor, node_modules, docker, testes, etc.)

Param(
    [string]$OutputFile = "gestor-editorial-producao.zip"
)

$ErrorActionPreference = "Stop"

Write-Host "═══════════════════════════════════════════════════════════" -ForegroundColor Cyan
Write-Host "  CRIANDO ZIP OTIMIZADO PARA PRODUÇÃO" -ForegroundColor Green
Write-Host "═══════════════════════════════════════════════════════════`n" -ForegroundColor Cyan

$rootDir = Split-Path -Parent $PSScriptRoot
$tempDir = Join-Path $env:TEMP "gestor-editorial-temp-$(Get-Random)"
$outputPath = Join-Path $rootDir $OutputFile

# Diretórios e arquivos a excluir
$excludePatterns = @(
    # Vendor e dependências (serão reinstalados)
    "backend\vendor",
    "backend\node_modules",
    
    # Docker (não necessário em produção)
    "docker",
    
    # Testes
    "backend\tests",
    
    # Git
    ".git",
    "backend\.git",
    
    # Arquivos de desenvolvimento
    "backend\.idea",
    "backend\.vscode",
    "backend\.fleet",
    "backend\.nova",
    "backend\.phpactor.json",
    "backend\.phpunit.result.cache",
    "backend\.phpunit.cache",
    "backend\Homestead.json",
    "backend\Homestead.yaml",
    
    # Logs e cache
    "backend\storage\logs\*.log",
    "backend\storage\framework\cache\*",
    "backend\storage\framework\sessions\*",
    "backend\storage\framework\views\*",
    "backend\storage\pail",
    "backend\storage\*.key",
    
    # Build temporário (será reconstruído)
    "backend\public\build",
    "backend\public\hot",
    "backend\public\storage",
    
    # Arquivos de ambiente (não devem ser enviados)
    "backend\.env",
    "backend\.env.backup",
    "backend\.env.production",
    
    # Arquivos do sistema
    "**\Thumbs.db",
    "**\.DS_Store",
    "**\*.log",
    
    # Cursor/IDE
    ".cursor",
    "**\.cursor",
    
    # Database SQLite local
    "backend\database\database.sqlite",
    
    # Scripts de desenvolvimento
    "backend\init-laravel.ps1"
)

Write-Host "[1] Criando diretório temporário..." -ForegroundColor Cyan
New-Item -ItemType Directory -Path $tempDir -Force | Out-Null
$projectDir = Join-Path $tempDir "gestor-editorial"
New-Item -ItemType Directory -Path $projectDir -Force | Out-Null

Write-Host "[2] Copiando arquivos necessários..." -ForegroundColor Cyan

# Copiar backend (excluindo padrões)
$backendSource = Join-Path $rootDir "backend"
$backendDest = Join-Path $projectDir "backend"

Write-Host "   Copiando backend..." -ForegroundColor Blue
robocopy $backendSource $backendDest /E /XD vendor node_modules tests .git .idea .vscode .fleet .nova /XF *.log .env .env.backup .env.production Thumbs.db .DS_Store /NFL /NDL /NJH /NJS | Out-Null

# Limpar storage (manter estrutura, remover conteúdo)
Write-Host "   Limpando storage..." -ForegroundColor Blue
$storageDirs = @(
    "storage\framework\cache",
    "storage\framework\sessions",
    "storage\framework\views",
    "storage\logs"
)
foreach ($dir in $storageDirs) {
    $fullPath = Join-Path $backendDest $dir
    if (Test-Path $fullPath) {
        Get-ChildItem -Path $fullPath -File | Remove-Item -Force -ErrorAction SilentlyContinue
    }
}

# Copiar instalador
Write-Host "   Copiando instalador..." -ForegroundColor Blue
$instaladorSource = Join-Path $rootDir "instalador"
$instaladorDest = Join-Path $projectDir "instalador"
Copy-Item -Path $instaladorSource -Destination $instaladorDest -Recurse -Force

# Copiar docs (opcional, mas útil)
Write-Host "   Copiando documentação..." -ForegroundColor Blue
$docsSource = Join-Path $rootDir "docs"
$docsDest = Join-Path $projectDir "docs"
Copy-Item -Path $docsSource -Destination $docsDest -Recurse -Force

# Copiar readme
Write-Host "   Copiando README..." -ForegroundColor Blue
Copy-Item -Path (Join-Path $rootDir "readme.md") -Destination (Join-Path $projectDir "readme.md") -Force -ErrorAction SilentlyContinue

Write-Host "[3] Criando arquivo ZIP..." -ForegroundColor Cyan

# Remover ZIP anterior se existir
if (Test-Path $outputPath) {
    Remove-Item $outputPath -Force
}

# Criar ZIP
Add-Type -AssemblyName System.IO.Compression.FileSystem
[System.IO.Compression.ZipFile]::CreateFromDirectory($projectDir, $outputPath, [System.IO.Compression.CompressionLevel]::Optimal, $false)

Write-Host "[4] Limpando diretório temporário..." -ForegroundColor Cyan
Remove-Item -Path $tempDir -Recurse -Force

# Calcular tamanho
$zipSize = (Get-Item $outputPath).Length
$zipSizeMB = [math]::Round($zipSize / 1MB, 2)

Write-Host "`n═══════════════════════════════════════════════════════════" -ForegroundColor Cyan
Write-Host "  ZIP CRIADO COM SUCESSO!" -ForegroundColor Green
Write-Host "═══════════════════════════════════════════════════════════`n" -ForegroundColor Cyan

Write-Host "Arquivo: $outputPath" -ForegroundColor Yellow
Write-Host "Tamanho: $zipSizeMB MB" -ForegroundColor Yellow
Write-Host "`nO que foi excluído:" -ForegroundColor Blue
Write-Host "  - vendor/ (será reinstalado via composer)" -ForegroundColor White
Write-Host "  - node_modules/ (será reinstalado via npm)" -ForegroundColor White
Write-Host "  - docker/ (não necessário em produção)" -ForegroundColor White
Write-Host "  - tests/ (testes não são necessários)" -ForegroundColor White
Write-Host "  - .git/ (controle de versão)" -ForegroundColor White
Write-Host "  - Logs e cache do storage/" -ForegroundColor White
Write-Host "  - Arquivos de desenvolvimento" -ForegroundColor White
Write-Host "`nPróximos passos:" -ForegroundColor Blue
Write-Host "  1. Faça upload do arquivo ZIP no filemanager" -ForegroundColor White
Write-Host "  2. Extraia o ZIP no servidor" -ForegroundColor White
Write-Host "  3. Execute: php instalador/instalar.php" -ForegroundColor White
Write-Host ""
