<?php
/**
 * Script de Instalação do Gestor Editorial
 * 
 * Este script automatiza a instalação do projeto Laravel após o upload via filemanager.
 * Execute este script após fazer o upload do projeto e configurar o .env
 * 
 * Pode ser executado via:
 * - Terminal: php instalador/instalar.php
 * - Navegador: https://seudominio.com/instalador/instalar.php
 */

// Detectar se está sendo executado via web ou CLI
$isWeb = php_sapi_name() !== 'cli';

// Se for web, iniciar output HTML
if ($isWeb) {
    header('Content-Type: text/html; charset=utf-8');
    echo '<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalador - Gestor Editorial</title>
    <style>
        body { font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif; background: #f5f5f5; padding: 20px; }
        .container { max-width: 900px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h1 { color: #2c3e50; border-bottom: 3px solid #3498db; padding-bottom: 10px; }
        .step { margin: 20px 0; padding: 15px; background: #f8f9fa; border-left: 4px solid #3498db; }
        .step-title { font-weight: bold; color: #3498db; font-size: 1.1em; }
        .success { color: #27ae60; }
        .error { color: #e74c3c; font-weight: bold; }
        .warning { color: #f39c12; }
        .info { color: #3498db; }
        pre { background: #2c3e50; color: #ecf0f1; padding: 15px; border-radius: 4px; overflow-x: auto; }
        .btn { display: inline-block; padding: 10px 20px; background: #3498db; color: white; text-decoration: none; border-radius: 4px; margin-top: 10px; }
        .btn:hover { background: #2980b9; }
        .loading { display: inline-block; width: 20px; height: 20px; border: 3px solid #f3f3f3; border-top: 3px solid #3498db; border-radius: 50%; animation: spin 1s linear infinite; }
        @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 Instalador do Gestor Editorial</h1>
';
    // Flush output para mostrar progresso em tempo real
    if (ob_get_level() > 0) {
        ob_end_flush();
    }
    ob_implicit_flush(true);
}

// Configurações
$rootDir = dirname(__DIR__);
$backendDir = $rootDir . '/backend';
$envFile = $backendDir . '/.env';
$envExampleFile = $backendDir . '/.env.example';

// #region agent log
$logFile = $rootDir . '/.cursor/debug.log';
$logData = [
    'sessionId' => 'installer-validation',
    'runId' => 'run1',
    'timestamp' => time() * 1000,
    'location' => 'instalar.php:13',
    'message' => 'Script iniciado',
    'data' => [
        'rootDir' => $rootDir,
        'backendDir' => $backendDir,
        'phpVersion' => phpversion(),
        'sapi' => php_sapi_name(),
        'hasReadline' => function_exists('readline'),
        'hasPosix' => function_exists('posix_isatty'),
        'hasExec' => function_exists('exec'),
    ],
    'hypothesisId' => 'A'
];
file_put_contents($logFile, json_encode($logData) . "\n", FILE_APPEND);
// #endregion

// Cores para output (se suportado)
$colors = [
    'reset' => "\033[0m",
    'red' => "\033[31m",
    'green' => "\033[32m",
    'yellow' => "\033[33m",
    'blue' => "\033[34m",
    'cyan' => "\033[36m",
];

function colorize($text, $color = 'reset') {
    global $colors, $isWeb;
    
    // Se for web, retornar texto sem cores (cores serão aplicadas via HTML)
    if ($isWeb) {
        return $text;
    }
    
    // Se não for terminal, retorna sem cores
    if (php_sapi_name() !== 'cli' || !function_exists('posix_isatty') || !posix_isatty(STDOUT)) {
        return $text;
    }
    return $colors[$color] . $text . $colors['reset'];
}

function printStep($step, $message) {
    global $isWeb;
    if ($isWeb) {
        echo "<div class='step'><div class='step-title'>[PASSO $step] $message</div></div>";
        flush();
    } else {
        echo colorize("\n[PASSO $step] ", 'cyan') . $message . "\n";
    }
}

function printSuccess($message) {
    global $isWeb;
    if ($isWeb) {
        echo "<div class='success'>✓ $message</div>";
        flush();
    } else {
        echo colorize("✓ ", 'green') . $message . "\n";
    }
}

function printError($message) {
    global $isWeb;
    if ($isWeb) {
        echo "<div class='error'>✗ ERRO: $message</div>";
        flush();
    } else {
        echo colorize("✗ ERRO: ", 'red') . $message . "\n";
    }
}

function printWarning($message) {
    global $isWeb;
    if ($isWeb) {
        echo "<div class='warning'>⚠ AVISO: $message</div>";
        flush();
    } else {
        echo colorize("⚠ AVISO: ", 'yellow') . $message . "\n";
    }
}

function printInfo($message) {
    global $isWeb;
    if ($isWeb) {
        echo "<div class='info'>ℹ $message</div>";
        flush();
    } else {
        echo colorize("ℹ ", 'blue') . $message . "\n";
    }
}

// Verificações iniciais
printStep(1, "Verificando pré-requisitos...");

// Verificar se estamos no diretório correto
if (!file_exists($backendDir)) {
    printError("Diretório 'backend' não encontrado!");
    printInfo("Execute este script a partir da raiz do projeto.");
    if ($isWeb) {
        echo "</div></body></html>";
    }
    exit(1);
}

// Verificar se o artisan existe
if (!file_exists($backendDir . '/artisan')) {
    printError("Arquivo 'artisan' não encontrado!");
    printInfo("Certifique-se de que o projeto Laravel foi extraído corretamente.");
    if ($isWeb) {
        echo "</div></body></html>";
    }
    exit(1);
}

// Verificar PHP
$phpVersion = phpversion();
printInfo("PHP versão: $phpVersion");
if (version_compare($phpVersion, '8.2.0', '<')) {
    printError("PHP 8.2 ou superior é necessário!");
    if ($isWeb) {
        echo "</div></body></html>";
    }
    exit(1);
}

// Verificar extensões necessárias
$requiredExtensions = ['pdo', 'pdo_mysql', 'mbstring', 'openssl', 'tokenizer', 'xml', 'ctype', 'json', 'fileinfo'];
$missingExtensions = [];
foreach ($requiredExtensions as $ext) {
    if (!extension_loaded($ext)) {
        $missingExtensions[] = $ext;
    }
}
if (!empty($missingExtensions)) {
    printError("Extensões PHP faltando: " . implode(', ', $missingExtensions));
    if ($isWeb) {
        echo "</div></body></html>";
    }
    exit(1);
}

printSuccess("Pré-requisitos verificados!");

// Verificar se deve continuar (parâmetro GET para web)
$continue = isset($_GET['continue']) && $_GET['continue'] == '1';

// Verificar/Criar arquivo .env
printStep(2, "Verificando arquivo .env...");

if (!file_exists($envFile)) {
    printWarning("Arquivo .env não encontrado!");
    
    if (file_exists($envExampleFile)) {
        printInfo("Criando .env a partir de .env.example...");
        if (!copy($envExampleFile, $envFile)) {
            printError("Não foi possível criar o arquivo .env!");
            if ($isWeb) {
                echo "</div></body></html>";
            }
            exit(1);
        }
        printSuccess(".env criado a partir de .env.example");
        printWarning("IMPORTANTE: Configure as variáveis de ambiente no arquivo .env antes de continuar!");
        printInfo("Especialmente: APP_KEY, DB_CONNECTION, DB_HOST, DB_DATABASE, DB_USERNAME, DB_PASSWORD");
        
        if ($isWeb && !$continue) {
            echo "<div class='warning'><strong>⚠ IMPORTANTE:</strong> Configure o arquivo .env antes de continuar!</div>";
            echo "<p>Após configurar o .env, <a href='?continue=1' class='btn'>Clique aqui para continuar</a></p>";
            echo "</div></body></html>";
            exit(0);
        } else if ($isWeb && $continue) {
            // Continuar após configurar .env
            printInfo("Continuando instalação...");
        } else {
            echo "\n";
            echo colorize("Pressione ENTER após configurar o .env para continuar...", 'yellow');
            if (function_exists('readline')) {
                readline();
            } else {
                fgets(STDIN);
            }
        }
    } else {
        printError("Arquivo .env.example não encontrado!");
        printInfo("Criando .env básico...");
        createBasicEnv($envFile);
        printWarning("IMPORTANTE: Configure o arquivo .env manualmente antes de continuar!");
        
        if ($isWeb && !$continue) {
            echo "<div class='warning'><strong>⚠ IMPORTANTE:</strong> Configure o arquivo .env antes de continuar!</div>";
            echo "<p>Após configurar o .env, <a href='?continue=1' class='btn'>Clique aqui para continuar</a></p>";
            echo "</div></body></html>";
            exit(0);
        } else if ($isWeb && $continue) {
            // Continuar após configurar .env
            printInfo("Continuando instalação...");
        } else {
            echo "\n";
            echo colorize("Pressione ENTER após configurar o .env para continuar...", 'yellow');
            if (function_exists('readline')) {
                readline();
            } else {
                fgets(STDIN);
            }
        }
    }
} else {
    printSuccess("Arquivo .env encontrado!");
    
    // Verificar se APP_KEY está configurado
    $envContent = file_get_contents($envFile);
    // #region agent log
    $logData = [
        'sessionId' => 'installer-validation',
        'runId' => 'run1',
        'timestamp' => time() * 1000,
        'location' => 'instalar.php:125',
        'message' => 'Verificando APP_KEY',
        'data' => [
            'hasAppKeyLine' => strpos($envContent, 'APP_KEY=') !== false,
            'hasBase64Key' => strpos($envContent, 'APP_KEY=base64:') !== false,
        ],
        'hypothesisId' => 'C'
    ];
    file_put_contents($logFile, json_encode($logData) . "\n", FILE_APPEND);
    // #endregion
    if (strpos($envContent, 'APP_KEY=') !== false) {
        $lines = explode("\n", $envContent);
        $hasKey = false;
        foreach ($lines as $line) {
            if (preg_match('/^APP_KEY=(.+)$/', $line, $matches)) {
                if (trim($matches[1]) !== '' && trim($matches[1]) !== 'base64:') {
                    $hasKey = true;
                    break;
                }
            }
        }
        if (!$hasKey) {
            printWarning("APP_KEY não está configurado. Será gerado automaticamente.");
        }
    }
}

// Verificar Composer
printStep(3, "Verificando Composer...");

// Verificar se exec() está disponível
if (!function_exists('exec')) {
    printError("Função exec() não está disponível!");
    printInfo("Entre em contato com o suporte da hospedagem para habilitar a função exec().");
    if ($isWeb) {
        echo "<div class='info'>Alternativa: Execute os comandos manualmente via SSH ou terminal.</div>";
        echo "</div></body></html>";
    }
    exit(1);
}

$composerPath = null;
$possiblePaths = [
    '/usr/local/bin/composer',
    '/usr/bin/composer',
    'composer', // No PATH
    $backendDir . '/vendor/bin/composer',
    '/opt/cpanel/composer/bin/composer', // cPanel
    '/usr/local/cpanel/3rdparty/bin/composer', // cPanel alternativo
];

printInfo("Procurando Composer nos caminhos padrão...");

foreach ($possiblePaths as $path) {
    if ($isWeb) {
        echo "<div class='info'>Verificando: $path</div>";
        flush();
    }
    
    if ($path === 'composer') {
        $output = [];
        $returnVar = 0;
        // Tentar com timeout
        @exec('composer --version 2>&1', $output, $returnVar);
        if ($returnVar === 0 && !empty($output)) {
            $composerPath = 'composer';
            if ($isWeb) {
                echo "<div class='success'>Composer encontrado no PATH: " . htmlspecialchars(implode(' ', $output)) . "</div>";
            }
            break;
        }
    } else {
        if (file_exists($path)) {
            if (is_executable($path)) {
                $composerPath = $path;
                if ($isWeb) {
                    echo "<div class='success'>Composer encontrado em: $path</div>";
                }
                break;
            } else {
                if ($isWeb) {
                    echo "<div class='warning'>Arquivo encontrado mas não executável: $path</div>";
                }
            }
        }
    }
}

if (!$composerPath) {
    printError("Composer não encontrado!");
    printInfo("O Composer pode não estar instalado ou não estar no PATH do sistema.");
    if ($isWeb) {
        echo "<div class='info'><strong>Opções:</strong></div>";
        echo "<ul>";
        echo "<li>Entre em contato com o suporte da hospedagem para instalar o Composer</li>";
        echo "<li>Ou execute manualmente via SSH: <code>composer install --no-dev --optimize-autoloader</code></li>";
        echo "<li>Ou baixe o Composer manualmente: <a href='https://getcomposer.org/download/' target='_blank'>https://getcomposer.org/download/</a></li>";
        echo "</ul>";
        echo "</div></body></html>";
    } else {
        printInfo("Instale o Composer: https://getcomposer.org/download/");
    }
    exit(1);
}

printSuccess("Composer encontrado: $composerPath");

// Instalar dependências do Composer
printStep(4, "Instalando dependências do Composer...");

// #region agent log
$logData = [
    'sessionId' => 'installer-validation',
    'runId' => 'run1',
    'timestamp' => time() * 1000,
    'location' => 'instalar.php:184',
    'message' => 'Antes de chdir e exec composer',
    'data' => [
        'currentDir' => getcwd(),
        'targetDir' => $backendDir,
        'composerPath' => $composerPath,
        'hasExec' => function_exists('exec'),
    ],
    'hypothesisId' => 'E'
];
file_put_contents($logFile, json_encode($logData) . "\n", FILE_APPEND);
// #endregion
chdir($backendDir);
$command = escapeshellarg($composerPath) . ' install --no-dev --optimize-autoloader 2>&1';
printInfo("Executando: $command");
$output = []; // Limpar output anterior
exec($command, $output, $returnVar);
// #region agent log
$logData = [
    'sessionId' => 'installer-validation',
    'runId' => 'run1',
    'timestamp' => time() * 1000,
    'location' => 'instalar.php:190',
    'message' => 'Após exec composer',
    'data' => [
        'returnVar' => $returnVar,
        'outputLines' => count($output),
        'currentDir' => getcwd(),
    ],
    'hypothesisId' => 'E'
];
file_put_contents($logFile, json_encode($logData) . "\n", FILE_APPEND);
// #endregion

if ($returnVar !== 0) {
    printError("Falha ao instalar dependências do Composer!");
    if ($isWeb) {
        echo "<pre>" . htmlspecialchars(implode("\n", $output)) . "</pre>";
    } else {
        echo implode("\n", $output) . "\n";
    }
    if ($isWeb) {
        echo "</div></body></html>";
    }
    exit(1);
}

printSuccess("Dependências do Composer instaladas!");

// Gerar APP_KEY se necessário
printStep(5, "Gerando chave da aplicação...");

// Verificar se APP_KEY já está configurada
$envContent = file_get_contents($envFile);
$needsKey = true;
if (preg_match('/^APP_KEY=(.+)$/m', $envContent, $matches)) {
    $keyValue = trim($matches[1]);
    if ($keyValue !== '' && $keyValue !== 'base64:' && strpos($keyValue, 'base64:') === 0) {
        $needsKey = false;
        printInfo("APP_KEY já está configurada.");
    }
}

if ($needsKey) {
    chdir($backendDir);
    
    // Verificar se artisan existe
    if (!file_exists('artisan')) {
        printError("Arquivo artisan não encontrado!");
        if ($isWeb) {
            echo "</div></body></html>";
        }
        exit(1);
    }
    
    $command = 'php artisan key:generate --force 2>&1';
    printInfo("Executando: $command");
    $output = []; // Limpar output anterior
    
    // Tentar gerar a chave
    exec($command, $output, $returnVar);
    
    if ($returnVar === 0) {
        printSuccess("Chave da aplicação gerada com sucesso!");
        
        // Verificar se a chave foi realmente adicionada ao .env
        $envContentAfter = file_get_contents($envFile);
        if (preg_match('/^APP_KEY=base64:(.+)$/m', $envContentAfter, $keyMatches)) {
            printInfo("APP_KEY configurada no arquivo .env");
        } else {
            printWarning("APP_KEY pode não ter sido salva corretamente. Verifique o arquivo .env");
        }
    } else {
        printWarning("Não foi possível gerar a APP_KEY usando artisan.");
        if ($isWeb) {
            echo "<pre>" . htmlspecialchars(implode("\n", $output)) . "</pre>";
        } else {
            echo implode("\n", $output) . "\n";
        }
        
        // Tentar gerar manualmente como fallback
        printInfo("Tentando gerar APP_KEY manualmente...");
        $key = 'base64:' . base64_encode(random_bytes(32));
        
        // Atualizar o .env
        $envContent = file_get_contents($envFile);
        if (preg_match('/^APP_KEY=.*$/m', $envContent)) {
            $envContent = preg_replace('/^APP_KEY=.*$/m', 'APP_KEY=' . $key, $envContent);
        } else {
            $envContent = "APP_KEY=$key\n" . $envContent;
        }
        
        if (file_put_contents($envFile, $envContent)) {
            printSuccess("APP_KEY gerada e configurada manualmente!");
        } else {
            printError("Não foi possível salvar a APP_KEY no arquivo .env!");
            if ($isWeb) {
                echo "<div class='warning'>";
                echo "<strong>Chave gerada (copie manualmente para o .env):</strong><br>";
                echo "<code style='word-break: break-all;'>APP_KEY=$key</code>";
                echo "</div>";
            } else {
                echo "\nChave gerada (copie manualmente para o .env):\n";
                echo "APP_KEY=$key\n";
            }
        }
    }
} else {
    printSuccess("APP_KEY já está configurada, pulando geração.");
}

// Executar migrações
printStep(6, "Executando migrações do banco de dados...");

$command = 'php artisan migrate --force 2>&1';
printInfo("Executando: $command");
$output = []; // Limpar output anterior
// #region agent log
$logData = [
    'sessionId' => 'installer-validation',
    'runId' => 'run1',
    'timestamp' => time() * 1000,
    'location' => 'instalar.php:216',
    'message' => 'Antes de executar migrate',
    'data' => ['command' => $command, 'currentDir' => getcwd()],
    'hypothesisId' => 'F'
];
file_put_contents($logFile, json_encode($logData) . "\n", FILE_APPEND);
// #endregion
exec($command, $output, $returnVar);
// #region agent log
$logData = [
    'sessionId' => 'installer-validation',
    'runId' => 'run1',
    'timestamp' => time() * 1000,
    'location' => 'instalar.php:220',
    'message' => 'Após migrate',
    'data' => ['returnVar' => $returnVar, 'outputLines' => count($output), 'hasError' => $returnVar !== 0],
    'hypothesisId' => 'F'
];
file_put_contents($logFile, json_encode($logData) . "\n", FILE_APPEND);
// #endregion

if ($returnVar !== 0) {
    printError("Falha ao executar migrações!");
    if ($isWeb) {
        echo "<pre>" . htmlspecialchars(implode("\n", $output)) . "</pre>";
    } else {
        echo implode("\n", $output) . "\n";
    }
    printInfo("Verifique as configurações do banco de dados no arquivo .env");
    if ($isWeb) {
        echo "</div></body></html>";
    }
    exit(1);
}

printSuccess("Migrações executadas com sucesso!");

// Executar seeders
printStep(7, "Executando seeders do banco de dados...");

$command = 'php artisan db:seed --force 2>&1';
printInfo("Executando: $command");
$output = []; // Limpar output anterior
exec($command, $output, $returnVar);
// #region agent log
$logData = [
    'sessionId' => 'installer-validation',
    'runId' => 'run1',
    'timestamp' => time() * 1000,
    'location' => 'instalar.php:234',
    'message' => 'Após db:seed',
    'data' => ['returnVar' => $returnVar, 'outputLines' => count($output)],
    'hypothesisId' => 'E'
];
file_put_contents($logFile, json_encode($logData) . "\n", FILE_APPEND);
// #endregion

if ($returnVar !== 0) {
    printWarning("Falha ao executar seeders!");
    if ($isWeb) {
        echo "<pre>" . htmlspecialchars(implode("\n", $output)) . "</pre>";
    } else {
        echo implode("\n", $output) . "\n";
    }
    printInfo("Você pode executar manualmente: php artisan db:seed");
} else {
    printSuccess("Seeders executados com sucesso!");
}

// Limpar e otimizar caches
printStep(8, "Limpando e otimizando caches...");

$commands = [
    'php artisan config:clear',
    'php artisan cache:clear',
    'php artisan route:clear',
    'php artisan view:clear',
    'php artisan config:cache',
    'php artisan route:cache',
    'php artisan view:cache',
];

foreach ($commands as $cmd) {
    printInfo("Executando: $cmd");
    $output = []; // Limpar output anterior
    exec($cmd . ' 2>&1', $output, $returnVar);
    // #region agent log
    $logData = [
        'sessionId' => 'installer-validation',
        'runId' => 'run1',
        'timestamp' => time() * 1000,
        'location' => 'instalar.php:259',
        'message' => 'Comando de cache executado',
        'data' => ['command' => $cmd, 'returnVar' => $returnVar],
        'hypothesisId' => 'E'
    ];
    file_put_contents($logFile, json_encode($logData) . "\n", FILE_APPEND);
    // #endregion
    if ($returnVar !== 0) {
        printWarning("Aviso ao executar: $cmd");
    }
}

printSuccess("Caches otimizados!");

// Verificar permissões de storage e bootstrap/cache
printStep(9, "Verificando permissões de diretórios...");

$directories = [
    $backendDir . '/storage',
    $backendDir . '/storage/app',
    $backendDir . '/storage/framework',
    $backendDir . '/storage/framework/cache',
    $backendDir . '/storage/framework/sessions',
    $backendDir . '/storage/framework/views',
    $backendDir . '/storage/logs',
    $backendDir . '/bootstrap/cache',
];

foreach ($directories as $dir) {
    if (file_exists($dir) && is_dir($dir)) {
        if (!is_writable($dir)) {
            printWarning("Diretório não tem permissão de escrita: $dir");
            printInfo("Execute: chmod -R 775 $dir");
            if (function_exists('chmod')) {
                @chmod($dir, 0775);
                printInfo("Tentativa de ajustar permissões executada.");
            }
        }
    }
}

printSuccess("Permissões verificadas!");

// Resumo final
echo "\n";
echo colorize("═══════════════════════════════════════════════════════════", 'cyan');
echo colorize("\n  INSTALAÇÃO CONCLUÍDA COM SUCESSO!", 'green');
echo colorize("\n═══════════════════════════════════════════════════════════\n", 'cyan');

printInfo("Próximos passos:");
echo "  1. Verifique se o servidor web está configurado corretamente\n";
echo "  2. Configure o DocumentRoot para apontar para: " . $backendDir . "/public\n";
echo "  3. Certifique-se de que o mod_rewrite está habilitado (Apache)\n";
echo "  4. Acesse a aplicação no navegador\n";
echo "\n";

printInfo("Credenciais padrão do sistema:");
if ($isWeb) {
    echo "<ul>";
    echo "<li><strong>Admin:</strong> admin@empatheia.me / admin123</li>";
    echo "<li><strong>Editor:</strong> editor@gestor-editorial.com / editor123</li>";
    echo "</ul>";
    echo "<div class='warning'><strong>⚠ ALTERE AS SENHAS PADRÃO APÓS O PRIMEIRO ACESSO!</strong></div>";
    echo "</div>";
    echo "<div style='margin-top: 20px;'><a href='../' class='btn'>Acessar Aplicação</a></div>";
    echo "</div></body></html>";
} else {
    echo "  Admin: admin@empatheia.me / admin123\n";
    echo "  Editor: editor@gestor-editorial.com / editor123\n";
    echo colorize("  ⚠ ALTERE AS SENHAS PADRÃO APÓS O PRIMEIRO ACESSO!", 'yellow');
    echo "\n";
    printSuccess("Instalação finalizada!");
}

function createBasicEnv($envFile) {
    $content = <<<'ENV'
APP_NAME="Gestor Editorial"
APP_ENV=production
APP_KEY=
APP_DEBUG=false
APP_TIMEZONE=UTC
APP_URL=http://localhost

APP_LOCALE=pt_BR
APP_FALLBACK_LOCALE=pt_BR
APP_FAKER_LOCALE=pt_BR

APP_MAINTENANCE_DRIVER=file
APP_MAINTENANCE_STORE=database

BCRYPT_ROUNDS=12

LOG_CHANNEL=stack
LOG_STACK=single
LOG_DEPRECATIONS_CHANNEL=null
LOG_LEVEL=debug

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=gestor_editorial
DB_USERNAME=root
DB_PASSWORD=

SESSION_DRIVER=database
SESSION_LIFETIME=120
SESSION_ENCRYPT=false
SESSION_PATH=/
SESSION_DOMAIN=null

BROADCAST_CONNECTION=log
FILESYSTEM_DISK=local
QUEUE_CONNECTION=database

CACHE_STORE=database
CACHE_PREFIX=

MEMCACHED_HOST=127.0.0.1

REDIS_CLIENT=phpredis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379

MAIL_MAILER=log
MAIL_HOST=127.0.0.1
MAIL_PORT=2525
MAIL_USERNAME=null
MAIL_PASSWORD=null
MAIL_ENCRYPTION=null
MAIL_FROM_ADDRESS="hello@example.com"
MAIL_FROM_NAME="${APP_NAME}"

AWS_ACCESS_KEY_ID=
AWS_SECRET_ACCESS_KEY=
AWS_DEFAULT_REGION=us-east-1
AWS_BUCKET=
AWS_USE_PATH_STYLE_ENDPOINT=false

VITE_APP_NAME="${APP_NAME}"
ENV;
    
    file_put_contents($envFile, $content);
}
