# Script PowerShell para instalação no Windows (desenvolvimento local)
# Para produção, use o instalar.php via SSH

Param(
    [switch]$SkipEnvCheck
)

$ErrorActionPreference = "Stop"

Write-Host "═══════════════════════════════════════════════════════════" -ForegroundColor Cyan
Write-Host "  INSTALADOR DO GESTOR EDITORIAL" -ForegroundColor Green
Write-Host "═══════════════════════════════════════════════════════════`n" -ForegroundColor Cyan

$rootDir = Split-Path -Parent $PSScriptRoot
$backendDir = Join-Path $rootDir "backend"
$envFile = Join-Path $backendDir ".env"
$envExampleFile = Join-Path $backendDir ".env.example"

# Verificar se o diretório backend existe
if (-not (Test-Path $backendDir)) {
    Write-Host "ERRO: Diretório 'backend' não encontrado!" -ForegroundColor Red
    Write-Host "Execute este script a partir da pasta 'instalador'" -ForegroundColor Yellow
    exit 1
}

# Verificar se o artisan existe
if (-not (Test-Path (Join-Path $backendDir "artisan"))) {
    Write-Host "ERRO: Arquivo 'artisan' não encontrado!" -ForegroundColor Red
    exit 1
}

Write-Host "[PASSO 1] Verificando pré-requisitos..." -ForegroundColor Cyan

# Verificar PHP
$phpVersion = & php -r "echo PHP_VERSION;" 2>$null
if (-not $phpVersion) {
    Write-Host "ERRO: PHP não encontrado no PATH!" -ForegroundColor Red
    Write-Host "Instale o PHP 8.2 ou superior" -ForegroundColor Yellow
    exit 1
}

Write-Host "✓ PHP encontrado: $phpVersion" -ForegroundColor Green

# Verificar Composer
$composerPath = Get-Command composer -ErrorAction SilentlyContinue
if (-not $composerPath) {
    Write-Host "ERRO: Composer não encontrado no PATH!" -ForegroundColor Red
    Write-Host "Instale o Composer: https://getcomposer.org/download/" -ForegroundColor Yellow
    exit 1
}

Write-Host "✓ Composer encontrado" -ForegroundColor Green

Write-Host "`n[PASSO 2] Verificando arquivo .env..." -ForegroundColor Cyan

# Verificar/Criar .env
if (-not (Test-Path $envFile)) {
    Write-Host "⚠ Arquivo .env não encontrado!" -ForegroundColor Yellow
    
    if (Test-Path $envExampleFile) {
        Write-Host "Criando .env a partir de .env.example..." -ForegroundColor Blue
        Copy-Item $envExampleFile $envFile
        Write-Host "✓ .env criado" -ForegroundColor Green
    } else {
        Write-Host "ERRO: Arquivo .env.example não encontrado!" -ForegroundColor Red
        exit 1
    }
    
    if (-not $SkipEnvCheck) {
        Write-Host "`n⚠ IMPORTANTE: Configure o arquivo .env antes de continuar!" -ForegroundColor Yellow
        Write-Host "Especialmente: APP_KEY, DB_CONNECTION, DB_HOST, DB_DATABASE, DB_USERNAME, DB_PASSWORD" -ForegroundColor Yellow
        Write-Host "`nPressione ENTER após configurar o .env para continuar..." -ForegroundColor Yellow
        Read-Host
    }
} else {
    Write-Host "✓ Arquivo .env encontrado" -ForegroundColor Green
}

Write-Host "`n[PASSO 3] Instalando dependências do Composer..." -ForegroundColor Cyan
Set-Location $backendDir
& composer install --no-dev --optimize-autoloader
if ($LASTEXITCODE -ne 0) {
    Write-Host "ERRO: Falha ao instalar dependências do Composer!" -ForegroundColor Red
    exit 1
}
Write-Host "✓ Dependências instaladas" -ForegroundColor Green

Write-Host "`n[PASSO 4] Gerando chave da aplicação..." -ForegroundColor Cyan
& php artisan key:generate --force
if ($LASTEXITCODE -ne 0) {
    Write-Host "⚠ Aviso: Não foi possível gerar a APP_KEY automaticamente" -ForegroundColor Yellow
} else {
    Write-Host "✓ Chave gerada" -ForegroundColor Green
}

Write-Host "`n[PASSO 5] Executando migrações..." -ForegroundColor Cyan
& php artisan migrate --force
if ($LASTEXITCODE -ne 0) {
    Write-Host "ERRO: Falha ao executar migrações!" -ForegroundColor Red
    Write-Host "Verifique as configurações do banco de dados no arquivo .env" -ForegroundColor Yellow
    exit 1
}
Write-Host "✓ Migrações executadas" -ForegroundColor Green

Write-Host "`n[PASSO 6] Executando seeders..." -ForegroundColor Cyan
& php artisan db:seed --force
if ($LASTEXITCODE -ne 0) {
    Write-Host "⚠ Aviso: Falha ao executar seeders" -ForegroundColor Yellow
} else {
    Write-Host "✓ Seeders executados" -ForegroundColor Green
}

Write-Host "`n[PASSO 7] Otimizando caches..." -ForegroundColor Cyan
& php artisan config:clear
& php artisan cache:clear
& php artisan route:clear
& php artisan view:clear
& php artisan config:cache
& php artisan route:cache
& php artisan view:cache
Write-Host "✓ Caches otimizados" -ForegroundColor Green

Write-Host "`n═══════════════════════════════════════════════════════════" -ForegroundColor Cyan
Write-Host "  INSTALAÇÃO CONCLUÍDA COM SUCESSO!" -ForegroundColor Green
Write-Host "═══════════════════════════════════════════════════════════`n" -ForegroundColor Cyan

Write-Host "Credenciais padrão do sistema:" -ForegroundColor Blue
Write-Host "  Admin: admin@empatheia.me / admin123" -ForegroundColor White
Write-Host "  Editor: editor@gestor-editorial.com / editor123" -ForegroundColor White
Write-Host "`n⚠ ALTERE AS SENHAS PADRÃO APÓS O PRIMEIRO ACESSO!" -ForegroundColor Yellow

Set-Location $rootDir
