<?php
/**
 * Script de Verificação de Pré-requisitos
 * Execute este script antes da instalação para verificar se tudo está pronto
 */

$rootDir = dirname(__DIR__);
$backendDir = $rootDir . '/backend';

$errors = [];
$warnings = [];
$success = [];

// Cores
$colors = [
    'reset' => "\033[0m",
    'red' => "\033[31m",
    'green' => "\033[32m",
    'yellow' => "\033[33m",
    'blue' => "\033[34m",
    'cyan' => "\033[36m",
];

function colorize($text, $color = 'reset') {
    global $colors;
    if (php_sapi_name() !== 'cli' || !function_exists('posix_isatty') || !posix_isatty(STDOUT)) {
        return $text;
    }
    return $colors[$color] . $text . $colors['reset'];
}

echo colorize("═══════════════════════════════════════════════════════════\n", 'cyan');
echo colorize("  VERIFICAÇÃO DE PRÉ-REQUISITOS - GESTOR EDITORIAL\n", 'green');
echo colorize("═══════════════════════════════════════════════════════════\n\n", 'cyan');

// 1. Verificar PHP
echo colorize("[1] Verificando PHP...\n", 'cyan');
$phpVersion = phpversion();
echo "   Versão: $phpVersion\n";
if (version_compare($phpVersion, '8.2.0', '>=')) {
    $success[] = "PHP $phpVersion (OK)";
    echo colorize("   ✓ PHP 8.2+ encontrado\n", 'green');
} else {
    $errors[] = "PHP 8.2+ necessário (encontrado: $phpVersion)";
    echo colorize("   ✗ PHP 8.2+ necessário\n", 'red');
}

// 2. Verificar extensões PHP
echo colorize("\n[2] Verificando extensões PHP...\n", 'cyan');
$requiredExtensions = [
    'pdo' => 'PDO',
    'pdo_mysql' => 'PDO MySQL',
    'mbstring' => 'Multibyte String',
    'openssl' => 'OpenSSL',
    'tokenizer' => 'Tokenizer',
    'xml' => 'XML',
    'ctype' => 'CTYPE',
    'json' => 'JSON',
    'fileinfo' => 'FileInfo',
];

foreach ($requiredExtensions as $ext => $name) {
    if (extension_loaded($ext)) {
        echo colorize("   ✓ $name\n", 'green');
        $success[] = "Extensão $name";
    } else {
        echo colorize("   ✗ $name faltando\n", 'red');
        $errors[] = "Extensão PHP faltando: $name";
    }
}

// 3. Verificar Composer
echo colorize("\n[3] Verificando Composer...\n", 'cyan');
$composerPath = null;
$possiblePaths = ['/usr/local/bin/composer', '/usr/bin/composer', 'composer'];

foreach ($possiblePaths as $path) {
    if ($path === 'composer') {
        $output = [];
        $returnVar = 0;
        @exec('composer --version 2>&1', $output, $returnVar);
        if ($returnVar === 0) {
            $composerPath = 'composer';
            $version = implode(' ', $output);
            echo "   $version\n";
            echo colorize("   ✓ Composer encontrado\n", 'green');
            $success[] = "Composer instalado";
            break;
        }
    } else {
        if (file_exists($path) && is_executable($path)) {
            $composerPath = $path;
            echo "   Caminho: $path\n";
            echo colorize("   ✓ Composer encontrado\n", 'green');
            $success[] = "Composer instalado";
            break;
        }
    }
}

if (!$composerPath) {
    echo colorize("   ✗ Composer não encontrado\n", 'red');
    $errors[] = "Composer não encontrado";
}

// 4. Verificar estrutura do projeto
echo colorize("\n[4] Verificando estrutura do projeto...\n", 'cyan');
if (file_exists($backendDir)) {
    echo colorize("   ✓ Diretório 'backend' encontrado\n", 'green');
    $success[] = "Estrutura do projeto";
} else {
    echo colorize("   ✗ Diretório 'backend' não encontrado\n", 'red');
    $errors[] = "Estrutura do projeto incompleta";
}

if (file_exists($backendDir . '/artisan')) {
    echo colorize("   ✓ Arquivo 'artisan' encontrado\n", 'green');
} else {
    echo colorize("   ✗ Arquivo 'artisan' não encontrado\n", 'red');
    $errors[] = "Arquivo artisan não encontrado";
}

if (file_exists($backendDir . '/composer.json')) {
    echo colorize("   ✓ Arquivo 'composer.json' encontrado\n", 'green');
} else {
    echo colorize("   ✗ Arquivo 'composer.json' não encontrado\n", 'red');
    $errors[] = "Arquivo composer.json não encontrado";
}

// 5. Verificar .env
echo colorize("\n[5] Verificando arquivo .env...\n", 'cyan');
$envFile = $backendDir . '/.env';
$envExampleFile = $backendDir . '/.env.example';

if (file_exists($envFile)) {
    echo colorize("   ✓ Arquivo .env encontrado\n", 'green');
    
    // Verificar se APP_KEY está configurado
    $envContent = file_get_contents($envFile);
    if (preg_match('/^APP_KEY=(.+)$/m', $envContent, $matches)) {
        if (trim($matches[1]) !== '' && trim($matches[1]) !== 'base64:') {
            echo colorize("   ✓ APP_KEY configurado\n", 'green');
        } else {
            echo colorize("   ⚠ APP_KEY não configurado\n", 'yellow');
            $warnings[] = "APP_KEY não configurado (será gerado automaticamente)";
        }
    }
} else {
    if (file_exists($envExampleFile)) {
        echo colorize("   ⚠ Arquivo .env não encontrado (será criado a partir de .env.example)\n", 'yellow');
        $warnings[] = "Arquivo .env não encontrado";
    } else {
        echo colorize("   ⚠ Arquivo .env.example não encontrado\n", 'yellow');
        $warnings[] = "Arquivo .env.example não encontrado";
    }
}

// 6. Verificar permissões
echo colorize("\n[6] Verificando permissões...\n", 'cyan');
$directories = [
    $backendDir . '/storage',
    $backendDir . '/bootstrap/cache',
];

foreach ($directories as $dir) {
    if (file_exists($dir)) {
        if (is_writable($dir)) {
            echo colorize("   ✓ $dir (gravável)\n", 'green');
        } else {
            echo colorize("   ⚠ $dir (sem permissão de escrita)\n", 'yellow');
            $warnings[] = "Permissão de escrita necessária: $dir";
        }
    } else {
        echo colorize("   ⚠ $dir (não existe)\n", 'yellow');
        $warnings[] = "Diretório não existe: $dir";
    }
}

// Resumo
echo "\n";
echo colorize("═══════════════════════════════════════════════════════════\n", 'cyan');
echo colorize("  RESUMO\n", 'green');
echo colorize("═══════════════════════════════════════════════════════════\n", 'cyan');

if (!empty($success)) {
    echo colorize("✓ Sucessos: " . count($success) . "\n", 'green');
}

if (!empty($warnings)) {
    echo colorize("⚠ Avisos: " . count($warnings) . "\n", 'yellow');
    foreach ($warnings as $warning) {
        echo "  - $warning\n";
    }
}

if (!empty($errors)) {
    echo colorize("✗ Erros: " . count($errors) . "\n", 'red');
    foreach ($errors as $error) {
        echo "  - $error\n";
    }
    echo "\n";
    echo colorize("Corrija os erros antes de prosseguir com a instalação!\n", 'red');
    exit(1);
} else {
    echo "\n";
    echo colorize("✓ Todos os pré-requisitos estão OK!\n", 'green');
    echo colorize("Você pode prosseguir com a instalação executando:\n", 'cyan');
    echo colorize("  php instalador/instalar.php\n\n", 'yellow');
    exit(0);
}
